import rclpy
from rclpy.action import ActionServer
from rclpy.node import Node
import RPi.GPIO as GPIO
import time

from tile_detector.action import MoveStepper

# 根据您的硬件连接定义GPIO引脚
EN, DIR, STEP = 17, 27, 23

class StepperActionServer(Node):

    def __init__(self):
        super().__init__('stepper_action_server')
        self.get_logger().info("步进电机Action Server已启动，等待指令...")

        GPIO.setmode(GPIO.BCM)
        GPIO.setwarnings(False)
        GPIO.setup([EN, DIR, STEP], GPIO.OUT, initial=GPIO.LOW)
        GPIO.output(EN, GPIO.LOW) # 使能电机

        self._action_server = ActionServer(
            self,
            MoveStepper,
            '/move_stepper',
            self.execute_callback)

    def execute_callback(self, goal_handle):
        steps_to_move = goal_handle.request.steps
        self.get_logger().info(f'收到移动指令: 移动 {steps_to_move} 步...')

        # 假设正数向右，负数向左
        direction = GPIO.HIGH if steps_to_move > 0 else GPIO.LOW
        GPIO.output(DIR, direction)

        delay = 0.001 # 速度

        feedback_msg = MoveStepper.Feedback()

        for i in range(abs(steps_to_move)):
            GPIO.output(STEP, GPIO.HIGH)
            time.sleep(delay)
            GPIO.output(STEP, GPIO.LOW)
            time.sleep(delay)

            # 发送反馈
            if i % 20 == 0: # 每20步发送一次反馈
                feedback_msg.steps_completed = i + 1
                goal_handle.publish_feedback(feedback_msg)

        goal_handle.succeed()

        result = MoveStepper.Result()
        result.success = True
        self.get_logger().info('移动完成！')
        return result

    def destroy_node(self):
        GPIO.output(EN, GPIO.HIGH) # 关闭电机使能
        GPIO.cleanup()
        super().destroy_node()

def main(args=None):
    rclpy.init(args=args)
    node = StepperActionServer()
    try:
        rclpy.spin(node)
    except KeyboardInterrupt:
        pass
    finally:
        node.destroy_node()
        rclpy.shutdown()

if __name__ == '__main__':
    main()