import rclpy
from rclpy.node import Node
from rclpy.action import ActionClient
import time

from robm_interfaces.msg import Frequency
from tile_detector.srv import Tap
from tile_detector.action import MoveStepper

# --- 参数配置 ---
STEPS_PER_TAP = 400       # 每次移动的步数 (间距)
TOTAL_TAPS = 10           # 总共敲击的次数
HOLLOW_FREQ_THRESHOLD = 1500.0 # 空鼓判断的频率阈值 (Hz) - !!需要根据实际测试调整!!

class MasterController(Node):

    def __init__(self):
        super().__init__('master_controller_node')

        # 创建 Subscriber
        self.freq_subscription = self.create_subscription(
            Frequency, '/audio/frequency', self.freq_callback, 10)

        # 创建 Service Client
        self.tap_cli = self.create_client(Tap, '/trigger_tap')
        while not self.tap_cli.wait_for_service(timeout_sec=1.0):
            self.get_logger().info('电磁铁服务不可用，等待中...')

        # 创建 Action Client
        self._action_client = ActionClient(self, MoveStepper, '/move_stepper')
        while not self._action_client.wait_for_server(timeout_sec=1.0):
            self.get_logger().info('步进电机服务不可用，等待中...')

        self.latest_freq = 0.0
        self.new_freq_received = False
        self.tap_count = 0

        self.get_logger().info('主控节点已初始化，准备开始检测...')
        time.sleep(1)
        self.run_detection_cycle()

    def freq_callback(self, msg):
        self.latest_freq = msg.frequency
        self.new_freq_received = True
        # self.get_logger().info(f'收到频率数据: {self.latest_freq:.2f} Hz')

    def call_tap_service(self):
        req = Tap.Request()
        self.future = self.tap_cli.call_async(req)
        rclpy.spin_until_future_complete(self, self.future)
        return self.future.result()

    def send_move_goal(self, steps):
        goal_msg = MoveStepper.Goal()
        goal_msg.steps = steps

        self.get_logger().info(f'正在发送移动目标: {steps} 步')
        self._send_goal_future = self._action_client.send_goal_async(goal_msg)
        self._send_goal_future.add_done_callback(self.goal_response_callback)

    def goal_response_callback(self, future):
        goal_handle = future.result()
        if not goal_handle.accepted:
            self.get_logger().info('移动目标被拒绝')
            return

        self._get_result_future = goal_handle.get_result_async()
        self._get_result_future.add_done_callback(self.get_result_callback)

    def get_result_callback(self, future):
        result = future.result().result
        if result.success:
            self.get_logger().info('移动完成，准备进行下一次敲击。')
            time.sleep(0.5)
            # 触发下一个循环
            self.run_detection_cycle()
        else:
            self.get_logger().error('移动失败！')


    def run_detection_cycle(self):
        if self.tap_count >= TOTAL_TAPS:
            self.get_logger().info('所有检测点完成，任务结束。')
            rclpy.shutdown()
            return

        # 1. 敲击
        self.get_logger().info(f"--- 检测点 {self.tap_count + 1}/{TOTAL_TAPS} ---")
        self.new_freq_received = False
        self.get_logger().info("正在发送敲击指令...")
        response = self.call_tap_service()
        if not response.success:
            self.get_logger().error("敲击失败，终止任务。")
            return

        # 2. 等待并获取频率数据
        timeout_start = time.time()
        while not self.new_freq_received and (time.time() - timeout_start) < 2.0:
             time.sleep(0.01)

        # 3. 分析数据
        if self.new_freq_received:
            self.get_logger().info(f"检测到主导频率: {self.latest_freq:.2f} Hz")
            if self.latest_freq > HOLLOW_FREQ_THRESHOLD:
                self.get_logger().warn(f"!!!!!! [结果]：检测到空鼓 !!!!!!")
            else:
                self.get_logger().info(f"[结果]：瓷砖坚实。")
        else:
            self.get_logger().error("超时：未收到有效的频率数据！")

        self.tap_count += 1

        # 4. 移动到下一个位置
        if self.tap_count < TOTAL_TAPS:
            self.send_move_goal(STEPS_PER_TAP)
        else:
            self.get_logger().info('所有检测点完成，任务结束。')
            rclpy.shutdown()

def main(args=None):
    rclpy.init(args=args)
    node = MasterController()
    rclpy.spin(node)
    node.destroy_node()
    rclpy.shutdown()

if __name__ == '__main__':
    main()