import queue
import subprocess
import threading
from typing import List

import rclpy
from rclpy.node import Node
from std_msgs.msg import String


def contains_cjk(text: str) -> bool:
    return any('\u4e00' <= char <= '\u9fff' for char in text)


class TextToSpeechNode(Node):
    """Subscribe to text and play synthesized speech through ALSA."""

    def __init__(self) -> None:
        super().__init__('text_to_speech_player')
        self.declare_parameter('device', 'hw:2,0')
        self.declare_parameter('english_voice', 'en-us')
        self.declare_parameter('chinese_voice', 'zh')
        self.declare_parameter('rate', 160)  # words per minute
        self.declare_parameter('amplitude', 150)  # 0-200

        self.device = self.get_parameter('device').get_parameter_value().string_value
        self.english_voice = self.get_parameter('english_voice').get_parameter_value().string_value
        self.chinese_voice = self.get_parameter('chinese_voice').get_parameter_value().string_value
        self.rate = int(self.get_parameter('rate').get_parameter_value().integer_value)
        self.amplitude = int(self.get_parameter('amplitude').get_parameter_value().integer_value)

        self._queue: queue.Queue[str] = queue.Queue()
        self._stop_event = threading.Event()
        self._worker = threading.Thread(target=self._worker_loop, daemon=True)
        self._worker.start()

        self.create_subscription(String, 'tts_input', self._on_text, 10)
        self.get_logger().info('TextToSpeechNode ready; publish text to topic tts_input')

    def destroy_node(self) -> bool:
        self._stop_event.set()
        if self._worker.is_alive():
            self._queue.put_nowait('')  # unblock
            self._worker.join(timeout=2.0)
        return super().destroy_node()

    def _on_text(self, msg: String) -> None:
        clean_text = msg.data.strip()
        if not clean_text:
            self.get_logger().debug('Received empty text message; ignoring')
            return
        self._queue.put(clean_text)

    def _worker_loop(self) -> None:
        while not self._stop_event.is_set():
            try:
                text = self._queue.get(timeout=0.2)
            except queue.Empty:
                continue

            if not text:
                continue

            voice = self.chinese_voice if contains_cjk(text) else self.english_voice
            self._speak(text, voice)
            self._queue.task_done()

    def _speak(self, text: str, voice: str) -> None:
        espeak_cmd = [
            'espeak',
            '-v',
            voice,
            '-s',
            str(self.rate),
            '-a',
            str(self.amplitude),
            '--stdout',
            text,
        ]
        aplay_cmd = [
            'aplay',
            '-q',
            '-D',
            self.device,
        ]

        try:
            espeak = subprocess.Popen(espeak_cmd, stdout=subprocess.PIPE, stderr=subprocess.PIPE)
        except FileNotFoundError:
            self.get_logger().error('espeak command not found; please install espeak-ng')
            return
        except Exception as exc:
            self.get_logger().error(f'Failed to start espeak: {exc}')
            return

        assert espeak.stdout is not None

        try:
            aplay = subprocess.Popen(aplay_cmd, stdin=espeak.stdout, stderr=subprocess.PIPE)
        except FileNotFoundError:
            self.get_logger().error('aplay command not found; please install alsa-utils')
            espeak.terminate()
            return
        except Exception as exc:
            self.get_logger().error(f'Failed to start aplay: {exc}')
            espeak.terminate()
            return

        # Detach espeak stdout once piping is set up to allow clean termination.
        espeak.stdout.close()

        espeak_stderr = ''
        aplay_stderr = ''

        try:
            _, espeak_err = espeak.communicate(timeout=30.0)
            espeak_stderr = espeak_err.decode('utf-8', errors='ignore') if espeak_err else ''
        except subprocess.TimeoutExpired:
            espeak.kill()
            self.get_logger().error('espeak timeout for text: %s', text)

        try:
            _, aplay_err = aplay.communicate(timeout=30.0)
            aplay_stderr = aplay_err.decode('utf-8', errors='ignore') if aplay_err else ''
        except subprocess.TimeoutExpired:
            aplay.kill()
            self.get_logger().error('aplay timeout for text: %s', text)

        if espeak.returncode not in (0, None):
            self.get_logger().error('espeak exited with code %s; stderr: %s', espeak.returncode, espeak_stderr)
        if aplay.returncode not in (0, None):
            self.get_logger().error('aplay exited with code %s; stderr: %s', aplay.returncode, aplay_stderr)


def main(args: List[str] | None = None) -> None:
    rclpy.init(args=args)
    node = TextToSpeechNode()
    try:
        rclpy.spin(node)
    except KeyboardInterrupt:
        pass
    finally:
        node.destroy_node()
        rclpy.shutdown()


if __name__ == '__main__':
    main()
