#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <sys/time.h>
#include <termios.h>
#include <fcntl.h>
#include <string.h>
#include <time.h>
#include "ACTS2200.h"

// Linux-friendly kbhit()
int kbhit(void) {
    struct termios oldt, newt;
    int ch;
    int oldf;
    tcgetattr(STDIN_FILENO, &oldt);
    newt = oldt;
    newt.c_lflag &= ~(ICANON | ECHO);
    tcsetattr(STDIN_FILENO, TCSANOW, &newt);
    oldf = fcntl(STDIN_FILENO, F_GETFL, 0);
    fcntl(STDIN_FILENO, F_SETFL, oldf | O_NONBLOCK);
    ch = getchar();
    tcsetattr(STDIN_FILENO, TCSANOW, &oldt);
    fcntl(STDIN_FILENO, F_SETFL, oldf);
    if (ch != EOF) {
        ungetc(ch, stdin);
        return 1;
    }
    return 0;
}

// now time (s)
double now_sec() {
    struct timeval tv;
    gettimeofday(&tv, NULL);
    return tv.tv_sec + tv.tv_usec / 1e6;
}

int main(void) {
    // ---------- USER SETTINGS ----------
    const double sampRate   = 10000.0;     // Hz
    const unsigned chunk    = 1000;        // points per read
    const unsigned aiChan   = 0;           // AI0
    const double  sens_mV_g = 500.0;       // accelerometer sensitivity (mV/g)
    const char   *save_dir  = "/home/pi/acc_data";
    // -----------------------------------

    HANDLE hDev;
    ACTS2200_AI_PARAM ai;
    double *buf = NULL;
    unsigned long long total = 0;
    double t0, t1;

    // Build output filename with timestamp
    char filename[512];
    time_t now = time(NULL);
    struct tm *lt = localtime(&now);
    if (!lt) { perror("localtime"); return -1; }
    snprintf(filename, sizeof(filename),
             "%s/ai0_accel_%04d%02d%02d_%02d%02d%02d.csv",
             save_dir,
             lt->tm_year + 1900, lt->tm_mon + 1, lt->tm_mday,
             lt->tm_hour, lt->tm_min, lt->tm_sec);

    printf("Opening device...\n");
    hDev = ACTS2200_DEV_Create(0);
    if (hDev == INVALID_HANDLE_VALUE) { printf("Open device failed\n"); return -1; }

    // Configure AI
    memset(&ai, 0, sizeof(ai));
    ai.nSampChanCount = 1;
    ai.CHParam[0].nChannel   = aiChan;
    ai.CHParam[0].nSampRange = ACTS2200_AI_SAMPRANGE_N10_P10V; // ±10V
    ai.CHParam[0].nRefGround = ACTS2200_AI_REFGND_RSE;         // single-ended
    ai.nSampMode     = ACTS2200_SAMPMODE_CONTINUOUS;
    ai.nSampsPerChan = 4096;
    ai.fSampRate     = sampRate;
    ai.nSampClkSrc   = ACTS2200_VAL_LOCAL;
    ai.StartTrig.nTrigType = ACTS2200_VAL_NONE;

    if (!ACTS2200_AI_InitTask(hDev, &ai)) { printf("AI_InitTask failed\n"); ACTS2200_DEV_Release(hDev); return -2; }
    if (!ACTS2200_AI_StartTask(hDev))     { printf("AI_StartTask failed\n"); ACTS2200_AI_ReleaseTask(hDev); ACTS2200_DEV_Release(hDev); return -3; }

    // open file (new file each run - write mode)
    FILE *fp = fopen(filename, "w");
    if (!fp) { printf("Cannot open %s\n", filename); ACTS2200_AI_StopTask(hDev); ACTS2200_AI_ReleaseTask(hDev); ACTS2200_DEV_Release(hDev); return -4; }
    fprintf(fp, "Time(s),Volt(V),Accel(g),Accel(m/s^2)\n");

    // compute scale
    const double sens_V_g = sens_mV_g / 1000.0; // V per g
    const double g_to_mps2 = 9.80665;

    buf = (double*)malloc(sizeof(double) * chunk);
    if (!buf) { printf("No memory\n"); fclose(fp); ACTS2200_AI_StopTask(hDev); ACTS2200_AI_ReleaseTask(hDev); ACTS2200_DEV_Release(hDev); return -5; }

    printf("Start continuous acquisition...\n");
    printf("Output file: %s\n", filename);
    printf("Sampling rate = %.1f Hz, chunk = %u points\n", sampRate, chunk);
    printf("Press any key to stop.\n");

    t0 = now_sec();
    while (!kbhit()) {
        unsigned got = 0;
        if (!ACTS2200_AI_ReadAnalog(hDev, buf, chunk, &got, 2.0, ACTS2200_FILLMODE_GroupByScanNumber)) {
            printf("Read timeout/error\n");
            continue;
        }
        t1 = now_sec();
        double elapsed = t1 - t0;
        double dt = 1.0 / sampRate;

        for (unsigned i = 0; i < got; ++i) {
            double t = elapsed + i * dt;
            double v = buf[i];
            double acc_g = v / sens_V_g;
            double acc_mps2 = acc_g * g_to_mps2;
            fprintf(fp, "%.6f,%.6f,%.6f,%.6f\n", t, v, acc_g, acc_mps2);
        }
        fflush(fp);
        total += got;
    }

    double T = now_sec() - t0;
    printf("\nStopping...\n");
    printf("Total points: %llu\n", total);
    printf("Total time: %.3f seconds\n", T);
    printf("Approx. effective rate: %.1f Hz\n", (T>0? total/T : 0.0));

    fclose(fp);
    free(buf);
    ACTS2200_AI_StopTask(hDev);
    ACTS2200_AI_ReleaseTask(hDev);
    ACTS2200_DEV_Release(hDev);
    printf("Done. Data saved to %s\n", filename);
    return 0;
}
