import atexit
from dataclasses import dataclass
from typing import Optional, Tuple


@dataclass
class DriverState:
    ready: bool = False
    error: Optional[str] = None
    last_cmd: Optional[Tuple[str, int]] = None  # (dir, speed)


class RosmasterWrapper:
    def __init__(self):
        self.ok = False
        self.err = None
        self.board = None
        try:
            from Rosmaster_Lib import Rosmaster  # type: ignore
            self.board = Rosmaster()
            try:
                self.board.create_receive_threading()
            except Exception:
                pass
            self.ok = True
        except Exception as e:
            self.err = str(e)
            self.ok = False

    def motion(self, vx=0.0, vy=0.0, vz=0.0):
        if not self.ok:
            raise RuntimeError(f"Rosmaster init failed: {self.err}")
        self.board.set_car_motion(vx, vy, vz)

    def stop(self):
        if self.ok:
            try:
                self.board.set_car_motion(0.0, 0.0, 0.0)
            except Exception:
                pass

    def cleanup(self):
        self.stop()


class DriverService:
    def __init__(self) -> None:
        self.state = DriverState()
        try:
            self._drv = RosmasterWrapper()
            self.state.ready = self._drv.ok
            self.state.error = None if self._drv.ok else self._drv.err
        except Exception as e:
            self._drv = None
            self.state.ready = False
            self.state.error = str(e)

        atexit.register(self.cleanup)

    def cleanup(self):
        try:
            if self._drv:
                self._drv.cleanup()
        except Exception:
            pass

    @staticmethod
    def _speed01(pct) -> float:
        try:
            s = max(0, min(100, int(pct)))
        except Exception:
            s = 70
        return round(s / 100.0, 3)

    def move(self, direction: str, speed_pct: int) -> Tuple[str, int]:
        direction = (direction or "").lower()
        sp01 = self._speed01(speed_pct)
        if not self.state.ready:
            raise RuntimeError(self.state.error or "driver not ready")

        if direction == "forward":
            self._drv.motion(vx=sp01, vy=0.0, vz=0.0)
        elif direction == "back":
            self._drv.motion(vx=-sp01, vy=0.0, vz=0.0)
        elif direction == "left":
            self._drv.motion(vx=0.0, vy=sp01, vz=0.0)
        elif direction == "right":
            self._drv.motion(vx=0.0, vy=-sp01, vz=0.0)
        elif direction == "stop":
            self._drv.stop()
        else:
            raise ValueError("unknown direction")

        self.state.last_cmd = (direction, int(speed_pct))
        return self.state.last_cmd

    def health(self):
        return {"ready": self.state.ready, "error": self.state.error}

    def set_servo(self, servo_id: int, angle: float) -> Tuple[int, float]:
        if not self.state.ready:
            raise RuntimeError(self.state.error or "driver not ready")
        # Try several common Rosmaster bus servo APIs
        angle = float(angle)
        impl = None
        if hasattr(self._drv.board, "set_bus_servo"):
            impl = getattr(self._drv.board, "set_bus_servo")
        elif hasattr(self._drv.board, "set_servo_angle"):
            impl = getattr(self._drv.board, "set_servo_angle")
        elif hasattr(self._drv.board, "bus_servo_set"):
            impl = getattr(self._drv.board, "bus_servo_set")
        if impl is None:
            raise RuntimeError("bus servo API not available on Rosmaster board")
        try:
            impl(int(servo_id), angle)
        except TypeError:
            # Some APIs expect ints only
            impl(int(servo_id), int(round(angle)))
        return int(servo_id), angle


DRIVER = DriverService()
