import math
from dataclasses import dataclass
from typing import Optional, Tuple


R_EARTH = 6378137.0  # meters


@dataclass
class Origin:
    lat: float
    lon: float
    alt: Optional[float] = None


class LocationService:
    def __init__(self) -> None:
        self._origin: Optional[Origin] = None

    def set_origin(self, lat: float, lon: float, alt: Optional[float] = None) -> Origin:
        self._origin = Origin(lat=lat, lon=lon, alt=alt)
        return self._origin

    def clear_origin(self) -> None:
        self._origin = None

    def get_origin(self) -> Optional[Origin]:
        return self._origin

    def xy_from_latlon(self, lat: float, lon: float) -> Tuple[Optional[float], Optional[float]]:
        if self._origin is None:
            return None, None
        lat0 = math.radians(self._origin.lat)
        lon0 = math.radians(self._origin.lon)
        lat1 = math.radians(lat)
        lon1 = math.radians(lon)
        dlat = lat1 - lat0
        dlon = lon1 - lon0
        x = R_EARTH * dlon * math.cos((lat0 + lat1) / 2.0)
        y = R_EARTH * dlat
        return x, y


LOCATION = LocationService()

