"""Lightweight reverse geocoding using BigDataCloud (no API key required).

Provides a simple helper to translate lat/lon into a human readable
location string in Chinese, e.g. "中华人民共和国江苏省南京市玄武区".
"""

from __future__ import annotations

import time
from typing import Optional, Tuple

import requests


_cache: dict[Tuple[int, int], tuple[float, str]] = {}


def _key(lat: float, lon: float) -> Tuple[int, int]:
    # Quantize to ~1e-4 degrees to limit cache cardinality (~11m)
    return (int(round(lat * 1e4)), int(round(lon * 1e4)))


def get_location(lat: float, lon: float, ttl_sec: int = 600) -> Optional[str]:
    """Resolve a human readable location string for lat/lon.

    - Uses an in-memory TTL cache to avoid frequent network calls.
    - Returns None on failure, callers should keep last known value.
    """
    try:
        k = _key(lat, lon)
        now = time.time()
        if k in _cache:
            ts, value = _cache[k]
            if now - ts < ttl_sec:
                return value

        url = (
            "https://api.bigdatacloud.net/data/reverse-geocode-client"
            f"?latitude={lat}&longitude={lon}&localityLanguage=zh"
        )
        resp = requests.get(url, timeout=3.5)
        resp.raise_for_status()
        data = resp.json() if resp.content else {}
        country = data.get("countryName", "") or ""
        province = data.get("principalSubdivision", "") or ""
        city = data.get("city", "") or ""
        district = data.get("locality", "") or ""
        # Combine parts. Strip spaces to look tidy.
        result = f"{country}{province}{city}{district}".strip()
        if not result:
            result = data.get("localityInfo", {}).get("informative", [{}])[-1].get("name", None)
        if result:
            _cache[k] = (now, result)
        return result
    except Exception:
        return None

