from __future__ import annotations

import math
from dataclasses import dataclass, asdict
from typing import Dict, Optional, Tuple


MM_PER_CM = 10.0
UNITS_PER_DEG = 1000.0 / 240.0  # ≈4.1667


@dataclass
class ServoConfig:
    servo_id: int
    zero: float
    min_pulse: int
    max_pulse: int
    label: str


@dataclass
class ArmPose:
    x_mm: float
    y_mm: float
    z_mm: float
    pitch_deg: float
    rotation_deg: float

    def as_cm(self) -> Dict[str, float]:
        return {
            "x": round(self.x_mm / MM_PER_CM, 3),
            "y": round(self.y_mm / MM_PER_CM, 3),
            "z": round(self.z_mm / MM_PER_CM, 3),
        }


class ArmKinematicsError(ValueError):
    """Raised when requested pose is invalid or unreachable."""


class ArmService:
    LENGTHS_MM = {"L1": 157.0, "L2": 291.0, "L3": 132.0, "L4": 180.0}
    MIN_RADIUS_MM = 40.0
    MAX_RADIUS_MM = LENGTHS_MM["L2"] + LENGTHS_MM["L3"] + LENGTHS_MM["L4"] - 35.0
    DEFAULT_PITCH_DEG = 80.0  # Ensures initial coord (5,0,5) is reachable
    KEYBOARD_STEP_MM = 4.0
    GRIPPER_STEP_PULSE = 25
    ROTATION_LIMIT = (-120.0, 120.0)

    def __init__(self) -> None:
        self.servo_cfg: Dict[int, ServoConfig] = {
            1: ServoConfig(1, 250, 0, 700, "基座"),
            2: ServoConfig(2, 500, 200, 800, "大臂"),
            3: ServoConfig(3, 368, 200, 1000, "小臂"),
            4: ServoConfig(4, 230, 0, 1000, "手腕"),
            5: ServoConfig(5, 512, 0, 1000, "手爪旋转"),
            6: ServoConfig(6, 520, 200, 900, "夹爪"),
        }
        self._board = None
        # 初始坐标 5cm, 0cm, 5cm (世界坐标)，默认俯仰 80°
        self.pose = ArmPose(
            x_mm=5.0 * MM_PER_CM,
            y_mm=0.0,
            z_mm=5.0 * MM_PER_CM,
            pitch_deg=self.DEFAULT_PITCH_DEG,
            rotation_deg=0.0,
        )
        self._gripper_value = 650  # 半开
        self._last_pulses: Dict[int, int] = {}

    # --- Public API -------------------------------------------------

    def attach_board(self, board) -> None:
        self._board = board

    def get_state(self) -> Dict[str, object]:
        return {
            "position_cm": self.pose.as_cm(),
            "pitch_deg": round(self.pose.pitch_deg, 2),
            "rotation_deg": round(self.pose.rotation_deg, 2),
            "gripper_pulse": int(self._gripper_value),
            "servos": dict(self._last_pulses),
        }

    def move_to(
        self,
        x_cm: float,
        y_cm: float,
        z_cm: float,
        pitch_deg: Optional[float] = None,
        rotation_deg: Optional[float] = None,
        move_time: float = 0.25,
    ) -> Dict[str, object]:
        solution = self._solve_pose(x_cm, y_cm, z_cm, pitch_deg)
        rotation = self._normalize_rotation(rotation_deg)
        pulses = solution["pulses"]
        pulses[5] = self._deg_to_pulse(5, rotation)
        pulses[6] = int(self._gripper_value)
        self._apply_servo_positions(pulses, move_time)
        self.pose = ArmPose(
            x_mm=solution["wrist"]["target_mm"][0],
            y_mm=solution["wrist"]["target_mm"][1],
            z_mm=solution["wrist"]["target_mm"][2],
            pitch_deg=solution["pitch_deg"],
            rotation_deg=rotation,
        )
        self._gripper_value = pulses[6]
        self._last_pulses = pulses.copy()
        return self.get_state()

    def jog(
        self, dx_mm: float = 0.0, dy_mm: float = 0.0, dz_mm: float = 0.0, move_time: float = 0.18
    ) -> Dict[str, object]:
        next_pose_cm = {
            "x": (self.pose.x_mm + dx_mm) / MM_PER_CM,
            "y": (self.pose.y_mm + dy_mm) / MM_PER_CM,
            "z": (self.pose.z_mm + dz_mm) / MM_PER_CM,
        }
        return self.move_to(
            next_pose_cm["x"],
            next_pose_cm["y"],
            next_pose_cm["z"],
            pitch_deg=self.pose.pitch_deg,
            rotation_deg=self.pose.rotation_deg,
            move_time=move_time,
        )

    def adjust_gripper(self, delta_units: int) -> Dict[str, object]:
        cfg = self.servo_cfg[6]
        new_value = self._gripper_value + delta_units * self.GRIPPER_STEP_PULSE
        new_value = max(cfg.min_pulse, min(cfg.max_pulse, int(round(new_value))))
        self._gripper_value = new_value
        self._send_simple(6, new_value)
        self._last_pulses[6] = new_value
        return self.get_state()

    def set_gripper(self, pulse_value: int) -> Dict[str, object]:
        cfg = self.servo_cfg[6]
        value = max(cfg.min_pulse, min(cfg.max_pulse, int(pulse_value)))
        self._gripper_value = value
        self._send_simple(6, value)
        self._last_pulses[6] = value
        return self.get_state()

    # --- Internal helpers ------------------------------------------

    def _normalize_rotation(self, rotation_deg: Optional[float]) -> float:
        if rotation_deg is None:
            return self.pose.rotation_deg
        low, high = self.ROTATION_LIMIT
        return max(low, min(high, float(rotation_deg)))

    def _solve_pose(
        self, x_cm: float, y_cm: float, z_cm: float, pitch_deg: Optional[float]
    ) -> Dict[str, object]:
        try:
            x_mm = float(x_cm) * MM_PER_CM
            y_mm = float(y_cm) * MM_PER_CM
            z_mm = float(z_cm) * MM_PER_CM
        except Exception as exc:  # noqa: BLE001
            raise ArmKinematicsError("坐标需要为数字") from exc
        pitch = self.pose.pitch_deg if pitch_deg is None else float(pitch_deg)
        pitch_rad = math.radians(pitch)
        planar = math.hypot(x_mm, y_mm)
        if planar < self.MIN_RADIUS_MM:
            raise ArmKinematicsError("目标过近，无法安全到达")
        base_angle = math.degrees(math.atan2(y_mm, x_mm))
        wrist_plane = planar - self.LENGTHS_MM["L4"] * math.cos(pitch_rad)
        wrist_vertical = (z_mm - self.LENGTHS_MM["L1"]) - self.LENGTHS_MM["L4"] * math.sin(pitch_rad)
        if wrist_plane < 0:
            raise ArmKinematicsError("目标过近，超出手爪行程")
        distance = math.hypot(wrist_plane, wrist_vertical)
        max_reach = self.LENGTHS_MM["L2"] + self.LENGTHS_MM["L3"] - 5.0
        min_reach = abs(self.LENGTHS_MM["L2"] - self.LENGTHS_MM["L3"]) + 5.0
        if distance > max_reach or distance < min_reach:
            raise ArmKinematicsError("超出机械臂可达范围")

        cos_elbow = self._clamp(
            (self.LENGTHS_MM["L2"] ** 2 + self.LENGTHS_MM["L3"] ** 2 - distance**2)
            / (2 * self.LENGTHS_MM["L2"] * self.LENGTHS_MM["L3"])
        )
        elbow = math.pi - math.acos(cos_elbow)

        cos_shoulder = self._clamp(
            (self.LENGTHS_MM["L2"] ** 2 + distance**2 - self.LENGTHS_MM["L3"] ** 2)
            / (2 * self.LENGTHS_MM["L2"] * distance)
        )
        shoulder = math.atan2(wrist_vertical, wrist_plane) + math.acos(cos_shoulder)
        wrist = pitch_rad - shoulder + elbow

        pulses = {
            1: self._deg_to_pulse(1, base_angle),
            2: self._deg_to_pulse(2, math.degrees(shoulder)),
            3: self._deg_to_pulse(3, math.degrees(elbow)),
            4: self._deg_to_pulse(4, math.degrees(wrist)),
        }
        return {
            "pulses": pulses,
            "angles_deg": {
                "base": round(base_angle, 3),
                "shoulder": round(math.degrees(shoulder), 3),
                "elbow": round(math.degrees(elbow), 3),
                "wrist": round(math.degrees(wrist), 3),
            },
            "pitch_deg": pitch,
            "wrist": {
                "target_mm": (x_mm, y_mm, z_mm),
                "planar_mm": planar,
                "distance": distance,
            },
        }

    def _deg_to_pulse(self, servo_id: int, degrees: float) -> int:
        cfg = self.servo_cfg[servo_id]
        pulse = cfg.zero + degrees * UNITS_PER_DEG
        if pulse < cfg.min_pulse - 1e-6 or pulse > cfg.max_pulse + 1e-6:
            raise ArmKinematicsError(f"{cfg.label} 超出安全范围")
        return int(round(max(cfg.min_pulse, min(cfg.max_pulse, pulse))))

    @staticmethod
    def _clamp(value: float, low: float = -1.0, high: float = 1.0) -> float:
        return max(low, min(high, value))

    def _ensure_board(self) -> None:
        if self._board is None:
            raise RuntimeError("机械臂控制板未连接")

    def _apply_servo_positions(self, pulses: Dict[int, int], move_time: float) -> None:
        self._ensure_board()
        move_time = max(0.05, float(move_time))
        payload = [[sid, int(val)] for sid, val in sorted(pulses.items())]
        self._board.bus_servo_set_position(move_time, payload)

    def _send_simple(self, servo_id: int, pulse: int, move_time: float = 0.15) -> None:
        self._ensure_board()
        self._board.bus_servo_set_position(move_time, [[int(servo_id), int(pulse)]])


ARM = ArmService()
