from flask import Flask, jsonify, render_template, request, send_from_directory, Response
from flask_cors import CORS
from flask_sock import Sock
import json
import os
import base64
from datetime import datetime
import subprocess
import time
from typing import Optional

import requests
from ros_robot_controller_sdk import Board

try:
    import cv2
except Exception:  # noqa: BLE001
    cv2 = None

# In a real project this would be: from services.sensor_service import SENSOR
from services.sensor_service import SENSOR
from services.driver_service import DRIVER
from services.location_service import LOCATION
from services.auto_service import AUTO
from services.music_service import MUSIC
from services.arm_service import ARM, ArmKinematicsError


app = Flask(__name__)
CORS(app, resources={r"/*": {"origins": "*"}})

sock = Sock(app)

BOARD_INIT_ERROR: Optional[Exception] = None
try:
    BOARD = Board()
    BOARD.enable_reception()
except Exception as exc:  # noqa: BLE001
    BOARD = None
    BOARD_INIT_ERROR = exc
    print(f"[servo-tools] Failed to initialize Board: {exc}")
else:
    try:
        MUSIC.attach_board(BOARD)
    except Exception:
        pass
    try:
        ARM.attach_board(BOARD)
    except Exception:
        pass

def _discover_electromagnet_agent_url():
    raw = (os.environ.get("ELECTROMAGNET_AGENT_URL") or "").strip()
    if raw:
        if raw.lower() in {"local", "direct", "none"}:
            return ""
        return raw.rstrip("/")
    default_url = "http://127.0.0.1:5500"
    try:
        resp = requests.get(
            default_url + "/health",
            timeout=float(os.environ.get("ELECTROMAGNET_AGENT_DISCOVERY_TIMEOUT", "0.25")),
        )
        if resp.ok:
            print(f"[electromagnet] detected agent at {default_url}")
        else:
            print(f"[electromagnet] agent at {default_url} responded with {resp.status_code}")
    except Exception as exc:
        print(f"[electromagnet] agent not reachable at {default_url}: {exc}")
    return default_url


ELECTROMAGNET_AGENT_URL = _discover_electromagnet_agent_url()
ELECTROMAGNET_AGENT_TIMEOUT = float(os.environ.get("ELECTROMAGNET_AGENT_TIMEOUT", "3.0"))

OutputDevice = None
electromagnet: Optional["OutputDevice"] = None

if not ELECTROMAGNET_AGENT_URL:
    try:
        from gpiozero import OutputDevice as _GpioOutputDevice
    except Exception:  # noqa: BLE001
        _GpioOutputDevice = None
    OutputDevice = _GpioOutputDevice

if not ELECTROMAGNET_AGENT_URL:
    default_pin = 14
    env_pin = os.environ.get("ELECTROMAGNET_PIN")
    try:
        ELECTROMAGNET_PIN = int(env_pin) if env_pin is not None else default_pin
    except Exception:
        ELECTROMAGNET_PIN = default_pin

    if OutputDevice is not None:
        try:
            electromagnet = OutputDevice(ELECTROMAGNET_PIN, active_high=True, initial_value=False)
        except Exception:
            electromagnet = None


# ---------- Helpers ----------

def ok(data, status=200):
    return jsonify({"ok": True, "data": data}), status


def err(code: str, message: str, details=None, status=400):
    return (
        jsonify({"ok": False, "error": {"code": code, "message": message, "details": details or {}}}),
        status,
    )

def voltage_percent(volt: float, full: float = 12.6, empty: float = 10.5):
    try:
        v = float(volt)
    except Exception:
        return None
    v = max(empty, min(full, v))
    pct = (v - empty) / (full - empty) * 100.0
    return round(pct, 1)


def _board_unavailable_response():
    message = "串口控制板不可用"
    if BOARD_INIT_ERROR is not None:
        message += f": {BOARD_INIT_ERROR}"
    return jsonify({"success": False, "message": message}), 500


def _electromagnet_provider():
    if ELECTROMAGNET_AGENT_URL:
        return "agent"
    if electromagnet is not None:
        return "local"
    return "none"



# ---------- Page ----------

@app.route("/")
def index():
    return render_template("index.html")


@app.route("/tools")
def servo_tools():
    return render_template("tools.html")


# Optional static aliases to support /asset or /assets paths
@app.get("/assets/<path:filename>")
def static_assets(filename: str):
    return app.send_static_file(filename)


@app.get("/asset/<path:filename>")
def static_asset(filename: str):
    return app.send_static_file(filename)


# ---------- Servo tools API ----------

def _parse_int(value, label: str, min_val: int, max_val: int):
    try:
        number = int(value)
    except (TypeError, ValueError) as exc:  # noqa: BLE001
        raise ValueError(f"{label} 必须是整数") from exc
    if number < min_val or number > max_val:
        raise ValueError(f"{label} 范围需在 {min_val}-{max_val}")
    return number


def _parse_float(value, label: str):
    try:
        return float(value)
    except (TypeError, ValueError) as exc:  # noqa: BLE001
        raise ValueError(f"{label} 需要为数字") from exc


@app.post("/api/set_id")
def api_set_id():
    if BOARD is None:
        return _board_unavailable_response()
    body = request.get_json(silent=True) or {}
    try:
        old_id = _parse_int(body.get("old_id"), "当前 ID", 0, 254)
        new_id = _parse_int(body.get("new_id"), "新 ID", 0, 253)
    except ValueError as exc:
        return jsonify({"success": False, "message": str(exc)}), 400
    try:
        BOARD.bus_servo_set_id(old_id, new_id)
        time.sleep(0.1)
        readback = BOARD.bus_servo_read_id(new_id)
        read_id = readback[0] if readback else None
        if read_id == new_id:
            return jsonify({"success": True, "message": "修改成功"})
        return jsonify({"success": False, "message": "验证失败，请重试"})
    except Exception as exc:  # noqa: BLE001
        return jsonify({"success": False, "message": f"操作失败: {exc}"}), 500


@app.post("/api/move")
def api_move():
    if BOARD is None:
        return _board_unavailable_response()
    body = request.get_json(silent=True) or {}
    try:
        servo_id = _parse_int(body.get("id"), "舵机 ID", 0, 253)
        position = _parse_int(body.get("pos"), "位置", 0, 10000)
    except ValueError as exc:
        return jsonify({"success": False, "message": str(exc)}), 400

    move_time = body.get("time", 0.1)
    try:
        move_time = float(move_time)
    except (TypeError, ValueError):
        return jsonify({"success": False, "message": "time 参数无效"}), 400
    move_time = max(0.02, move_time)

    try:
        BOARD.bus_servo_set_position(move_time, [[servo_id, position]])
    except Exception as exc:  # noqa: BLE001
        return jsonify({"success": False, "message": f"移动失败: {exc}"}), 500
    return jsonify({"success": True, "message": "操作完成"})


@app.get("/api/arm/state")
def api_arm_state():
    try:
        data = ARM.get_state()
    except Exception as exc:  # noqa: BLE001
        return jsonify({"success": False, "message": f"获取失败: {exc}"}), 500
    return jsonify({"success": True, "data": data})


def _ensure_arm_available():
    if BOARD is None:
        raise RuntimeError("board_unavailable")


def _arm_response(data):
    return jsonify({"success": True, "data": data})


@app.post("/api/arm/move")
def api_arm_move():
    try:
        _ensure_arm_available()
    except RuntimeError:
        return _board_unavailable_response()
    body = request.get_json(silent=True) or {}
    state = ARM.get_state()
    pos = state.get("position_cm", {}) if isinstance(state, dict) else {}
    default_x = pos.get("x", 5.0)
    default_y = pos.get("y", 0.0)
    default_z = pos.get("z", 5.0)
    try:
        x = _parse_float(body.get("x", default_x), "X")
        y = _parse_float(body.get("y", default_y), "Y")
        z = _parse_float(body.get("z", default_z), "Z")
        pitch = body.get("pitch")
        rotation = body.get("rotation")
        pitch_val = None if pitch is None else _parse_float(pitch, "俯仰角")
        rotation_val = None if rotation is None else _parse_float(rotation, "旋转角")
        move_time = float(body.get("time", 0.25))
        data = ARM.move_to(x, y, z, pitch_deg=pitch_val, rotation_deg=rotation_val, move_time=move_time)
        return _arm_response(data)
    except ArmKinematicsError as exc:
        return jsonify({"success": False, "message": str(exc)}), 400
    except ValueError as exc:
        return jsonify({"success": False, "message": str(exc)}), 400
    except Exception as exc:  # noqa: BLE001
        return jsonify({"success": False, "message": f"机械臂指令失败: {exc}"}), 500


@app.post("/api/arm/jog")
def api_arm_jog():
    try:
        _ensure_arm_available()
    except RuntimeError:
        return _board_unavailable_response()
    body = request.get_json(silent=True) or {}
    try:
        dx = float(body.get("dx", 0.0))
        dy = float(body.get("dy", 0.0))
        dz = float(body.get("dz", 0.0))
        move_time = float(body.get("time", 0.18))
        data = ARM.jog(dx_mm=dx, dy_mm=dy, dz_mm=dz, move_time=move_time)
        return _arm_response(data)
    except ArmKinematicsError as exc:
        return jsonify({"success": False, "message": str(exc)}), 400
    except Exception as exc:  # noqa: BLE001
        return jsonify({"success": False, "message": f"机械臂微调失败: {exc}"}), 500


@app.post("/api/arm/gripper")
def api_arm_gripper():
    try:
        _ensure_arm_available()
    except RuntimeError:
        return _board_unavailable_response()
    body = request.get_json(silent=True) or {}
    mode = (body.get("mode") or "").lower()
    try:
        if mode == "set":
            value = _parse_int(body.get("value"), "夹爪位置", 0, 1000)
            data = ARM.set_gripper(value)
        else:
            delta = body.get("delta", 0)
            delta_int = int(delta)
            if delta_int == 0:
                raise ValueError("delta 需要非零整数")
            data = ARM.adjust_gripper(delta_int)
        return _arm_response(data)
    except ValueError as exc:
        return jsonify({"success": False, "message": str(exc)}), 400
    except Exception as exc:  # noqa: BLE001
        return jsonify({"success": False, "message": f"夹爪控制失败: {exc}"}), 500


def _run_system_command(cmd):
    try:
        proc = subprocess.run(
            cmd,
            capture_output=True,
            text=True,
            timeout=float(os.environ.get("TOOLS_COMMAND_TIMEOUT", "30")),
        )
        return proc.returncode, proc.stdout.strip(), proc.stderr.strip()
    except FileNotFoundError as exc:  # noqa: BLE001
        return 127, "", str(exc)
    except subprocess.TimeoutExpired:
        return 124, "", "命令执行超时"
    except Exception as exc:  # noqa: BLE001
        return 1, "", str(exc)


def _read_net_bytes():
    stats = {}
    try:
        with open("/proc/net/dev", "r", encoding="utf-8") as fp:
            for line in fp.read().splitlines()[2:]:
                if not line or ":" not in line:
                    continue
                iface, data = line.split(":", 1)
                iface = iface.strip()
                if iface == "lo":
                    continue
                parts = data.split()
                if len(parts) < 8:
                    continue
                rx = int(parts[0])
                tx = int(parts[8])
                stats[iface] = {"rx": rx, "tx": tx}
    except Exception:
        return {}
    return stats


def _sample_bandwidth(interval=1.0):
    start = _read_net_bytes()
    if not start:
        return "无法读取 /proc/net/dev"
    time.sleep(max(0.2, interval))
    end = _read_net_bytes()
    if not end:
        return "无法读取 /proc/net/dev"
    lines = []
    total_rx = 0
    total_tx = 0
    for iface, first in start.items():
        if iface not in end:
            continue
        delta_rx = max(0, end[iface]["rx"] - first["rx"])
        delta_tx = max(0, end[iface]["tx"] - first["tx"])
        total_rx += delta_rx
        total_tx += delta_tx
        down = delta_rx / interval
        up = delta_tx / interval
        lines.append(f"{iface}: ↓ {down/1024:.2f} KB/s ↑ {up/1024:.2f} KB/s")
    if not lines:
        return "无可用网卡数据"
    lines.append(f"汇总: ↓ {total_rx/interval/1024:.2f} KB/s ↑ {total_tx/interval/1024:.2f} KB/s")
    return "\n".join(lines)


SYSTEM_ACTIONS = {
    "restart_mainpanel": {
        "label": "重启 mainpanel.service",
        "command": ["sudo", "systemctl", "restart", "mainpanel.service"],
    },
    "restart_seg": {
        "label": "重启 seg.service",
        "command": ["sudo", "systemctl", "restart", "seg.service"],
    },
    "reboot_host": {
        "label": "重启主机",
        "command": ["sudo", "reboot"],
    },
    "show_ip": {
        "label": "查看 IP (ip a)",
        "command": ["bash", "-lc", "ip -br addr"],
    },
    "net_speed": {
        "label": "实时网速",
        "command": None,
    },
}


@app.post("/api/tools/run")
def tools_run():
    body = request.get_json(silent=True) or {}
    action = (body.get("action") or "").strip()
    info = SYSTEM_ACTIONS.get(action)
    if not info:
        return jsonify({"success": False, "message": "未知操作"}), 400
    if action == "net_speed":
        output = _sample_bandwidth(interval=float(body.get("interval", 1.0) or 1.0))
        return jsonify({"success": True, "action": action, "output": output})

    code, stdout_text, stderr_text = _run_system_command(info["command"])
    success = code == 0
    message = stdout_text or stderr_text or f"命令返回码 {code}"
    return (
        jsonify(
            {
                "success": success,
                "action": action,
                "output": message,
                "code": code,
            }
        ),
        200 if success else 500,
    )


@app.post("/api/driver/beep")
def driver_beep():
    body = request.get_json(silent=True) or {}
    count = body.get("count", 5)
    duration = body.get("duration_ms", 80)
    gap = body.get("gap_s", 0.05)
    try:
        DRIVER.beep(times=count, duration_ms=duration, gap_s=gap)
    except Exception as exc:  # noqa: BLE001
        return jsonify({"success": False, "message": f"蜂鸣器调用失败: {exc}"}), 500
    return jsonify({"success": True, "message": "蜂鸣器提示已播放"})


@app.get("/api/music/list")
def music_list():
    try:
        tracks = MUSIC.list_tracks()
    except Exception as exc:  # noqa: BLE001
        return jsonify({"success": False, "message": str(exc)}), 500
    return jsonify({"success": True, "tracks": tracks})


@app.post("/api/music/note")
def music_note():
    body = request.get_json(silent=True) or {}
    note = (body.get("note") or "").strip()
    duration = float(body.get("duration", 0.3) or 0.3)
    try:
        MUSIC.play_named(note, duration_s=max(0.1, duration))
        return jsonify({"success": True, "message": f"播放 {note}"})
    except Exception as exc:  # noqa: BLE001
        return jsonify({"success": False, "message": str(exc)}), 500


@app.post("/api/music/song")
def music_song():
    body = request.get_json(silent=True) or {}
    track = (body.get("track") or "").strip()
    tempo = float(body.get("tempo", 0.35) or 0.35)
    if not track:
        return jsonify({"success": False, "message": "缺少曲目名称"}), 400
    try:
        MUSIC.play_file(track, tempo_s=max(0.1, tempo))
        return jsonify({"success": True, "message": f"乐曲 {track} 播放完成"})
    except Exception as exc:  # noqa: BLE001
        return jsonify({"success": False, "message": str(exc)}), 500


@app.post("/snapshot")
def snapshot():
    try:
        body = request.get_json(silent=True) or {}
        data_url = body.get("image")
        if not data_url or not isinstance(data_url, str) or ";base64," not in data_url:
            return err("BAD_REQUEST", "missing or invalid image data URL")
        suffix = str(body.get("suffix") or "").strip()
        if suffix:
            suffix = "_" + "".join(ch for ch in suffix if ch.isalnum() or ch in "-_")[:6]

        header, b64 = data_url.split(",", 1)
        # default to png
        ext = "png"
        if header.startswith("data:image/"):
            ext = header.split("/")[1].split(";")[0] or "png"
        img_bytes = base64.b64decode(b64)

        os.makedirs("file/img", exist_ok=True)
        ts = datetime.now().strftime("%Y%m%d_%H%M%S")
        fname = f"seubot_{ts}{suffix}." + ext
        fpath = os.path.join("file", "img", fname)
        with open(fpath, "wb") as f:
            f.write(img_bytes)
        return ok({"path": fpath, "filename": fname})
    except Exception as e:  # noqa: BLE001
        return err("SNAPSHOT_FAILED", str(e), status=500)


@app.get("/snapshot/list")
def snapshot_list():
    try:
        base = os.path.join("file", "img")
        if not os.path.isdir(base):
            return ok({"items": []})
        items = []
        for name in os.listdir(base):
            if not name.lower().endswith((".png", ".jpg", ".jpeg")):
                continue
            path = os.path.join(base, name)
            try:
                st = os.stat(path)
                items.append({
                    "filename": name,
                    "url": f"/snapshot/file/{name}",
                    "mtime": int(st.st_mtime),
                })
            except OSError:
                continue
        # newest first
        items.sort(key=lambda x: x["mtime"], reverse=True)
        return ok({"items": items})
    except Exception as e:  # noqa: BLE001
        return err("SNAPSHOT_LIST_FAILED", str(e), status=500)


@app.get("/snapshot/file/<path:filename>")
def snapshot_file(filename: str):
    # serve saved snapshots
    base = os.path.join("file", "img")
    return send_from_directory(base, filename)


@app.delete("/snapshot/file/<path:filename>")
def snapshot_delete(filename: str):
    try:
        base = os.path.join("file", "img")
        path = os.path.join(base, filename)
        if not os.path.isfile(path):
            return err("NOT_FOUND", "file not found", status=404)
        os.remove(path)
        return ok({"deleted": filename})
    except Exception as e:  # noqa: BLE001
        return err("SNAPSHOT_DELETE_FAILED", str(e), status=500)


# (license endpoints removed per request)


# ---------- Capture control ----------

@app.post("/start_capture")
def start_capture():
    try:
        body = request.get_json(silent=True) or {}
        rate_hz = body.get("rate_hz")
        csv_path = SENSOR.start(rate_hz=rate_hz)
        return ok({"csv": csv_path, "capturing": True, "rate_hz": SENSOR.state.rate_hz})
    except Exception as e:  # noqa: BLE001 - return structured error
        return err("START_FAILED", str(e), status=500)


@app.post("/stop_capture")
def stop_capture():
    try:
        csv_path = SENSOR.stop()
        return ok({"csv": csv_path, "capturing": False})
    except Exception as e:  # noqa: BLE001 - return structured error
        return err("STOP_FAILED", str(e), status=500)


# ---------- Electromagnet ----------

def _agent_request(path: str, method: str = "GET", payload=None):
    if not ELECTROMAGNET_AGENT_URL:
        raise RuntimeError("agent URL not configured")
    url = ELECTROMAGNET_AGENT_URL + path
    try:
        resp = requests.request(method, url, json=payload, timeout=ELECTROMAGNET_AGENT_TIMEOUT)
    except Exception as exc:  # noqa: BLE001
        raise RuntimeError(str(exc)) from exc
    try:
        data = resp.json()
    except Exception as exc:  # noqa: BLE001
        raise RuntimeError(f"agent response parse error: {exc}") from exc
    if resp.status_code >= 400 or not data.get("ok"):
        err_msg = data.get("error") if isinstance(data, dict) else "agent error"
        raise RuntimeError(err_msg)
    return data


def _electromagnet_state():
    if ELECTROMAGNET_AGENT_URL:
        try:
            data = _agent_request("/electromagnet/state")
            return data.get("state")
        except Exception:
            return None
    if electromagnet is None:
        return None
    try:
        return "on" if bool(electromagnet.value) else "off"
    except Exception:
        return None


@app.get("/electromagnet/state")
def electromagnet_state():
    state = _electromagnet_state()
    available = state is not None
    provider = _electromagnet_provider()
    return ok({"state": state or "unknown", "available": available, "provider": provider})


@app.post("/electromagnet/activate")
def electromagnet_activate():
    try:
        body = request.get_json(silent=True) or {}
    except Exception:
        body = {}
    state = (body.get("state") or "").strip().lower()
    if state not in {"on", "off"}:
        return err("BAD_REQUEST", "invalid state, must be 'on' or 'off'", status=400)
    if ELECTROMAGNET_AGENT_URL:
        try:
            data = _agent_request("/electromagnet/activate", method="POST", payload={"state": state})
            return ok({"state": data.get("state"), "via": "agent", "provider": "agent"})
        except Exception as e:  # noqa: BLE001
            return err("ELECTROMAGNET_AGENT_FAILED", str(e), status=502)
    if electromagnet is None:
        return err("ELECTROMAGNET_UNAVAILABLE", "gpiozero not available or pin init failed", status=500)
    try:
        if state == "on":
            electromagnet.on()
        else:
            electromagnet.off()
        return ok({"state": state, "via": "local", "provider": "local"})
    except Exception as e:  # noqa: BLE001
        return err("ELECTROMAGNET_FAILED", str(e), status=500)


# ---------- Location / XY ----------

@app.post("/location/origin")
def set_origin():
    try:
        body = request.get_json(silent=True) or {}
        lat = float(body.get("lat"))
        lon = float(body.get("lon"))
        alt = body.get("alt")
        alt_f = float(alt) if alt is not None else None
        o = LOCATION.set_origin(lat, lon, alt_f)
        return ok({"lat": o.lat, "lon": o.lon, "alt": o.alt})
    except Exception as e:  # noqa: BLE001
        return err("ORIGIN_SET_FAILED", str(e), status=400)


@app.delete("/location/origin")
def clear_origin():
    LOCATION.clear_origin()
    return ok({"cleared": True})


@app.get("/location/xy")
def current_xy():
    try:
        lat = SENSOR.state.gps_lat
        lon = SENSOR.state.gps_lon
        x, y = LOCATION.xy_from_latlon(lat, lon)
        return ok({"x": x, "y": y, "z": SENSOR.state.gps_alt_m})
    except Exception as e:  # noqa: BLE001
        return err("XY_FAILED", str(e), status=500)


# ---------- SLAM / unit detection ----------

@app.post("/slam/start")
def slam_start():
    # placeholder - real impl would start backend job
    return ok({"running": True})


@app.post("/slam/done")
def slam_done():
    try:
        body = request.get_json(silent=True) or {}
        a = float(body.get("a", AUTO.unit.a))
        b = float(body.get("b", AUTO.unit.b))
        u = AUTO.set_unit(a, b)
        return ok({"a": u.a, "b": u.b})
    except Exception as e:  # noqa: BLE001
        return err("SLAM_DONE_FAILED", str(e), status=400)


# ---------- Auto planner ----------

@app.post("/auto/plan")
def auto_plan():
    try:
        body = request.get_json(silent=True) or {}
        nx = int(body.get("nx", 1))
        ny = int(body.get("ny", 1))
        mode = (body.get("mode") or "serpentine").lower()
        start = (body.get("start") or "tl").lower()
        pts = AUTO.plan_path(nx, ny, mode, start)
        return ok({"plan": pts, "count": len(pts), "unit": {"a": AUTO.unit.a, "b": AUTO.unit.b}})
    except Exception as e:  # noqa: BLE001
        return err("AUTO_PLAN_FAILED", str(e), status=400)


# ---------- Video feed (usb0 MJPEG) ----------


def _resolve_resolution(res: str):
    res_map = {
        "360p": (640, 360),
        "480p": (848, 480),
        "720p": (1280, 720),
        "1080p": (1920, 1080),
        "2k": (2560, 1440),
        "1440p": (2560, 1440),
    }
    return res_map.get((res or "").lower(), res_map["720p"])



def _usb_stream(device: str = "/dev/video0", res: str = "720p"):
    if cv2 is None:
        raise RuntimeError("OpenCV not available")
    width, height = _resolve_resolution(res)
    cap = cv2.VideoCapture(device)
    if not cap.isOpened():
        raise RuntimeError(f"failed to open camera {device}")
    cap.set(cv2.CAP_PROP_FRAME_WIDTH, width)
    cap.set(cv2.CAP_PROP_FRAME_HEIGHT, height)
    try:
        while True:
            ok, frame = cap.read()
            if not ok:
                break
            ok, buf = cv2.imencode(".jpg", frame, [int(cv2.IMWRITE_JPEG_QUALITY), 80])
            if not ok:
                continue
            chunk = buf.tobytes()
            yield (b"--frame\r\n"
                   b"Content-Type: image/jpeg\r\n\r\n" + chunk + b"\r\n")
    finally:
        cap.release()


@app.get("/usb<int:cam>")
def usb_cam_feed(cam: int):
    res = request.args.get("res", "720p")
    device = f"/dev/video{cam}"
    try:
        return Response(_usb_stream(device, res), mimetype="multipart/x-mixed-replace; boundary=frame")
    except Exception as e:  # noqa: BLE001
        return err("VIDEO_FEED_FAILED", str(e), status=503)


@app.get("/usb0")
def usb0_feed():
    res = request.args.get("res", "720p")
    try:
        return Response(_usb_stream("/dev/video0", res), mimetype="multipart/x-mixed-replace; boundary=frame")
    except Exception as e:  # noqa: BLE001
        return err("VIDEO_FEED_FAILED", str(e), status=503)


@app.get("/video_feed")
def video_feed():
    # Legacy alias; points to /usb0
    return usb0_feed()


# ---------- WebSockets ----------

@sock.route("/ws")
def ws_samples(ws):
    import time as _t

    # Emit samples only when capturing. Honor configured rate when active.
    while True:
        if SENSOR.is_capturing():
            sample = SENSOR.next_sample()
            if sample is not None:
                ws.send(json.dumps(sample))
            # sleep based on configured rate_hz
            try:
                hz = max(1, int(SENSOR.state.rate_hz))
            except Exception:  # defensive fallback
                hz = 50
            _t.sleep(1.0 / float(hz))
        else:
            # idle: push status (GPS/temperature/location) at low rate
            try:
                status = {
                    "t": _t.time(),
                    "temp_c": SENSOR.state.temperature_c,
                    "gps": {
                        "lat": SENSOR.state.gps_lat,
                        "lon": SENSOR.state.gps_lon,
                        "alt_m": SENSOR.state.gps_alt_m,
                        "sats": SENSOR.state.gps_sats,
                    },
                    "location": SENSOR.state.location_str,
                }
                ws.send(json.dumps(status))
            except Exception:
                pass
            _t.sleep(1.0)


@sock.route("/log_ws")
def ws_logs(ws):
    import time as _t

    while True:
        now = _t.strftime("%H:%M:%S", _t.localtime())
        if SENSOR.is_capturing() and SENSOR.get_csv_path():
            ws.send(f"[{now}] writing → {SENSOR.get_csv_path()}")
        else:
            ws.send(f"[{now}] idle")
        _t.sleep(1)


# ---------- Driver control ----------

@app.post("/driver/move")
def driver_move():
    try:
        body = request.get_json(silent=True) or {}
        direction = (body.get("dir") or "").lower()
        speed = int(body.get("speed", 70))
        d, s = DRIVER.move(direction, speed)
        return ok({"dir": d, "speed": s})
    except Exception as e:  # noqa: BLE001
        return err("DRIVE_FAILED", str(e), status=400)

@app.post("/move")
def driver_move_root():
    # Compatibility alias for mecanum drive control
    return driver_move()




@app.get("/driver/health")
def driver_health():
    try:
        h = DRIVER.health()
        return ok(h)
    except Exception as e:  # noqa: BLE001
        return err("HEALTH_FAILED", str(e), status=500)


@app.post("/drive/start")
def drive_start():
    try:
        body = request.get_json(silent=True) or {}
        target = body.get("target") or {}
        # Placeholder: in real impl, navigate to absolute XY target
        # Start capture once reached target (simulated immediately)
        SENSOR.start()
        return ok({"navigating": True, "target": target, "csv": SENSOR.state.csv_path})
    except Exception as e:  # noqa: BLE001
        return err("DRIVE_START_FAILED", str(e), status=400)


@app.post("/drive/work")
def drive_work():
    try:
        body = request.get_json(silent=True) or {}
        point = body.get("point") or {}
        # Placeholder: real impl should navigate and perform work/knock
        # Here we just acknowledge the command
        return ok({"arrived": True, "point": point, "action": "knock"})
    except Exception as e:  # noqa: BLE001
        return err("DRIVE_WORK_FAILED", str(e), status=400)


@app.post("/driver/servo")
def driver_servo():
    try:
        body = request.get_json(silent=True) or {}
        servo_id = int(body.get("id"))
        angle = float(body.get("angle"))
        sid, ang = DRIVER.set_servo(servo_id, angle)
        return ok({"id": sid, "angle": ang})
    except Exception as e:  # noqa: BLE001
        return err("SERVO_FAILED", str(e), status=400)


# ---------- Device / Battery ----------

@app.get("/device/battery")
def device_battery():
    try:
        volt = DRIVER.battery_voltage()
    except Exception as e:  # noqa: BLE001
        return err("BATTERY_READ_FAILED", str(e), status=503)
    pct = voltage_percent(volt)
    warn = pct is not None and pct <= 20
    data = {"volt": volt, "percent": pct, "warn": warn, "warn_threshold_percent": 20,
            "full_volt": 12.6, "protect_volt": 10.5}
    if warn:
        data["note"] = "低电量，建议尽快充电"
    return ok(data)


# ---------- Errors ----------

@app.errorhandler(404)
def not_found(_):
    return err("NOT_FOUND", "route not found", status=404)


if __name__ == "__main__":
    # For production: hypercorn app:app --bind 0.0.0.0:5000
    app.run(host="0.0.0.0", port=50000, debug=True)
