#!/usr/bin/env python3
"""
Standalone electromagnet control agent.

Expose simple HTTP API so other processes (e.g. mainpanel running in venv)
can control GPIO via native environment permissions.
"""

from __future__ import annotations

import argparse
import json
import os
import sys
from http import HTTPStatus
from http.server import BaseHTTPRequestHandler, HTTPServer
from socketserver import ThreadingMixIn
from typing import Optional

try:
    from gpiozero import OutputDevice
except Exception:  # noqa: BLE001
    OutputDevice = None


class Electromagnet:
    def __init__(self, pin: int, active_high: bool = True):
        self.pin = pin
        self.active_high = active_high
        self._device: Optional[OutputDevice] = None
        if OutputDevice is None:
            raise RuntimeError("gpiozero not available, install it in the native env")
        self._device = OutputDevice(pin, active_high=active_high, initial_value=False)

    @property
    def is_on(self) -> bool:
        if self._device is None:
            return False
        try:
            return bool(self._device.value)
        except Exception as exc:  # noqa: BLE001
            raise RuntimeError(str(exc)) from exc

    def set_state(self, state: str) -> str:
        if self._device is None:
            raise RuntimeError("electromagnet not initialized")
        if state == "on":
            self._device.on()
        else:
            self._device.off()
        return state


class ThreadedHTTPServer(ThreadingMixIn, HTTPServer):
    daemon_threads = True


class AgentHandler(BaseHTTPRequestHandler):
    electromagnet: Optional[Electromagnet] = None

    def _send_json(self, status: HTTPStatus, payload: dict):
        data = json.dumps(payload).encode("utf-8")
        self.send_response(status.value)
        self.send_header("Content-Type", "application/json")
        self.send_header("Content-Length", str(len(data)))
        self.end_headers()
        self.wfile.write(data)

    def _read_json(self) -> dict:
        length = int(self.headers.get("Content-Length", "0"))
        raw = self.rfile.read(length) if length > 0 else b""
        if not raw:
            return {}
        try:
            return json.loads(raw.decode("utf-8"))
        except json.JSONDecodeError as exc:
            raise ValueError("invalid JSON body") from exc

    def log_message(self, format: str, *args):  # noqa: A003
        sys.stderr.write("%s - - [%s] %s\n" % (self.client_address[0], self.log_date_time_string(), format % args))

    def do_GET(self):  # noqa: N802
        if self.path == "/health":
            self._send_json(HTTPStatus.OK, {"ok": True})
            return
        if self.path == "/electromagnet/state":
            em = self.__class__.electromagnet
            if em is None:
                self._send_json(HTTPStatus.SERVICE_UNAVAILABLE, {"ok": False, "error": "not initialized"})
                return
            try:
                state = "on" if em.is_on else "off"
            except Exception as exc:  # noqa: BLE001
                self._send_json(HTTPStatus.INTERNAL_SERVER_ERROR, {"ok": False, "error": str(exc)})
                return
            self._send_json(HTTPStatus.OK, {"ok": True, "state": state, "pin": em.pin})
            return
        self._send_json(HTTPStatus.NOT_FOUND, {"ok": False, "error": "not found"})

    def do_POST(self):  # noqa: N802
        if self.path == "/electromagnet/activate":
            em = self.__class__.electromagnet
            if em is None:
                self._send_json(HTTPStatus.SERVICE_UNAVAILABLE, {"ok": False, "error": "not initialized"})
                return
            try:
                body = self._read_json()
            except ValueError as exc:
                self._send_json(HTTPStatus.BAD_REQUEST, {"ok": False, "error": str(exc)})
                return
            state = (body.get("state") or "").strip().lower()
            if state not in {"on", "off"}:
                self._send_json(HTTPStatus.BAD_REQUEST, {"ok": False, "error": "state must be 'on' or 'off'"})
                return
            try:
                em.set_state(state)
            except Exception as exc:  # noqa: BLE001
                self._send_json(HTTPStatus.INTERNAL_SERVER_ERROR, {"ok": False, "error": str(exc)})
                return
            self._send_json(HTTPStatus.OK, {"ok": True, "state": state, "pin": em.pin})
            return
        self._send_json(HTTPStatus.NOT_FOUND, {"ok": False, "error": "not found"})


def main():
    parser = argparse.ArgumentParser(description="Electromagnet GPIO agent")
    parser.add_argument("--host", default="0.0.0.0", help="bind address (default: 0.0.0.0)")
    parser.add_argument("--port", type=int, default=5500, help="listening port (default: 5500)")
    parser.add_argument("--pin", type=int, default=int(os.environ.get("AGENT_ELECTROMAGNET_PIN", 14)),
                        help="BCM pin number (default: env AGENT_ELECTROMAGNET_PIN or 14)")
    args = parser.parse_args()

    try:
        electromagnet = Electromagnet(args.pin)
    except Exception as exc:  # noqa: BLE001
        sys.stderr.write(f"Failed to init electromagnet on BCM{args.pin}: {exc}\n")
        sys.exit(1)

    AgentHandler.electromagnet = electromagnet
    server = ThreadedHTTPServer((args.host, args.port), AgentHandler)
    print(f"Electromagnet agent running on {args.host}:{args.port} (BCM{args.pin})")
    try:
        server.serve_forever()
    except KeyboardInterrupt:
        print("\nStopping agent...")
    finally:
        server.server_close()


if __name__ == "__main__":
    main()
