import time
import os
import csv
import random
import threading
from dataclasses import dataclass, field
from typing import Optional
import threading

from .gps import get_location


@dataclass
class CaptureState:
    capturing: bool = False
    csv_path: Optional[str] = None
    rate_hz: int = 50  # default 50Hz; configurable via /start_capture
    last_sample: Optional[dict] = field(default=None)
    # writer resources
    _writer_thread: Optional[threading.Thread] = field(default=None, repr=False)
    _stop_event: Optional[threading.Event] = field(default=None, repr=False)
    _csv_file: Optional[object] = field(default=None, repr=False)
    _csv_writer: Optional[csv.writer] = field(default=None, repr=False)
    # Simulated environment status
    temperature_c: float = 25.0
    gps_lat: float = 31.2304   # example: Shanghai
    gps_lon: float = 121.4737
    gps_alt_m: float = 10.0
    gps_sats: int = 8
    location_str: Optional[str] = None
    last_geocode_ts: float = 0.0


class SensorService:
    def __init__(self) -> None:
        self.state = CaptureState()
        self._lock = threading.Lock()
        self._geocode_inflight = False

    def _schedule_geocode_if_needed(self) -> None:
        """Kick a background geocode if stale; non-blocking to sample loop."""
        try:
            with self._lock:
                now = time.time()
                # Only attempt if at least 30s since last try
                if self._geocode_inflight or (now - self.state.last_geocode_ts) < 30:
                    return
                lat, lon = self.state.gps_lat, self.state.gps_lon
                self._geocode_inflight = True
                self.state.last_geocode_ts = now

            def _worker():
                try:
                    loc = get_location(lat, lon)
                    if loc:
                        with self._lock:
                            self.state.location_str = loc
                finally:
                    with self._lock:
                        self._geocode_inflight = False

            t = threading.Thread(target=_worker, daemon=True)
            t.start()
        except Exception:
            # Fail silently; next cycle may retry
            with self._lock:
                self._geocode_inflight = False

    def _choose_csv_path(self) -> str:
        base = os.environ.get("CSV_BASE_DIR", "/sta/data/1")
        ts = int(time.time())
        fname = f"{ts}.csv"
        # attempt to create; fallback to workspace path if not permitted
        try:
            os.makedirs(base, exist_ok=True)
            return os.path.join(base, fname)
        except Exception:
            local_base = os.path.join("file", "data", "1")
            os.makedirs(local_base, exist_ok=True)
            return os.path.join(local_base, fname)

    def _open_csv(self, path: str):
        f = open(path, "w", newline="", encoding="utf-8")
        w = csv.writer(f)
        w.writerow([
            "t", "volt", "accel_g", "accel_mps2",
            "temp_c", "gps_lat", "gps_lon", "gps_alt_m", "gps_sats", "location",
        ])
        f.flush()
        return f, w

    def start(self, rate_hz: Optional[int] = None) -> str:
        with self._lock:
            if rate_hz:
                self.state.rate_hz = max(1, int(rate_hz))
            if self.state.capturing:
                return self.state.csv_path or self._choose_csv_path()
            path = self._choose_csv_path()
            f, w = self._open_csv(path)
            self.state.csv_path = path
            self.state.capturing = True
            self.state._stop_event = threading.Event()
            self.state._csv_file = f
            self.state._csv_writer = w

            t = threading.Thread(target=self._writer_loop, name="sensor-writer", daemon=True)
            self.state._writer_thread = t
            t.start()
            return path

    def stop(self) -> Optional[str]:
        with self._lock:
            self.state.capturing = False
            if self.state._stop_event:
                self.state._stop_event.set()
            t = self.state._writer_thread
            f = self.state._csv_file
            self.state._writer_thread = None
            self.state._stop_event = None
            self.state._csv_writer = None
            self.state._csv_file = None
        # join outside lock to avoid deadlocks
        try:
            if t:
                t.join(timeout=2.0)
        except Exception:
            pass
        try:
            if f:
                f.close()
        except Exception:
            pass
        with self._lock:
            return self.state.csv_path

    def is_capturing(self) -> bool:
        with self._lock:
            return self.state.capturing

    def get_csv_path(self) -> Optional[str]:
        with self._lock:
            return self.state.csv_path

    def _generate_sample(self) -> dict:
        t = time.time()
        volt = 2.5 + random.uniform(-0.05, 0.05)
        accel_g = random.uniform(-0.02, 0.02)
        accel_mps2 = accel_g * 9.80665

        with self._lock:
            self.state.temperature_c += random.uniform(-0.02, 0.03)
            self.state.temperature_c = max(-40.0, min(125.0, self.state.temperature_c))
            self.state.gps_lat += random.uniform(-1e-6, 1e-6)
            self.state.gps_lon += random.uniform(-1e-6, 1e-6)
            self.state.gps_alt_m += random.uniform(-0.02, 0.02)
            if random.random() < 0.05:
                self.state.gps_sats = max(0, min(20, self.state.gps_sats + random.choice([-1, 0, 1])))

        self._schedule_geocode_if_needed()

        with self._lock:
            sample = {
                "t": t,
                "volt": volt,
                "accel_g": accel_g,
                "accel_mps2": accel_mps2,
                "temp_c": self.state.temperature_c,
                "gps": {
                    "lat": self.state.gps_lat,
                    "lon": self.state.gps_lon,
                    "alt_m": self.state.gps_alt_m,
                    "sats": self.state.gps_sats,
                },
                "location": self.state.location_str,
            }
            self.state.last_sample = sample
            return sample

    def _writer_loop(self):
        while True:
            with self._lock:
                stop = self.state._stop_event.is_set() if self.state._stop_event else True
                rate = max(1, int(self.state.rate_hz))
                writer = self.state._csv_writer
                f = self.state._csv_file
            if stop:
                break
            sample = self._generate_sample()
            # write csv row
            try:
                if writer and f:
                    writer.writerow([
                        sample.get("t"), sample.get("volt"), sample.get("accel_g"), sample.get("accel_mps2"),
                        sample.get("temp_c"),
                        sample.get("gps", {}).get("lat"), sample.get("gps", {}).get("lon"),
                        sample.get("gps", {}).get("alt_m"), sample.get("gps", {}).get("sats"),
                        sample.get("location"),
                    ])
                    f.flush()
            except Exception:
                pass
            time.sleep(1.0 / float(rate))

    def next_sample(self) -> Optional[dict]:
        with self._lock:
            if not self.state.capturing:
                return None
            return self.state.last_sample


# Simple singleton (can be swapped for DI later)
SENSOR = SensorService()
