import os
import threading
import time
from typing import List, Optional


class MusicService:
    NOTE_ALIASES = {
        "do": 1,
        "di": 2,
        "reb": 2,
        "re": 3,
        "ri": 4,
        "mib": 4,
        "mi": 5,
        "fa": 6,
        "fi": 7,
        "solb": 7,
        "sol": 8,
        "si": 9,
        "lab": 9,
        "la": 10,
        "li": 11,
        "sib": 11,
        "ti": 12,
        "si#": 12,
    }
    NOTE_INDEX = {
        1: 0,
        2: 1,
        3: 2,
        4: 3,
        5: 4,
        6: 5,
        7: 6,
        8: 7,
        9: 8,
        10: 9,
        11: 10,
        12: 11,
    }

    def __init__(self, board=None, base_freq: float = 261.63):
        self.board = board
        self.base_freq = base_freq
        self._lock = threading.Lock()

    def _ensure_board(self):
        if self.board is None:
            raise RuntimeError("控制板未就绪，无法播放蜂鸣器")

    def attach_board(self, board):
        self.board = board

    def list_tracks(self) -> List[str]:
        base = os.path.join("file", "music")
        if not os.path.isdir(base):
            return []
        tracks = []
        for name in os.listdir(base):
            if not name.lower().endswith(".txt"):
                continue
            tracks.append(os.path.splitext(name)[0])
        return sorted(tracks, key=lambda x: (len(x), x))

    def note_name_to_value(self, note: str) -> Optional[int]:
        name = (note or "").strip().lower()
        if not name:
            return None
        if name.isdigit():
            return int(name)
        return self.NOTE_ALIASES.get(name)

    def symbol_to_value(self, symbol: str) -> Optional[tuple]:
        raw = symbol.strip()
        if not raw:
            return None
        sustain = 0
        while len(raw) > 1 and raw.endswith("-"):
            sustain += 1
            raw = raw[:-1]
        if raw in {"|", "."}:
            return ("rest", sustain + 1)
        if raw in {"0", "rest"}:
            return ("rest", sustain + 1)

        octave_shift = 0
        # handle high octave marks (multiple ')
        while raw.endswith("'"):
            octave_shift += 1
            raw = raw[:-1]

        if not raw:
            return ("rest", sustain + 1)

        if raw.startswith("-"):
            octave_shift -= 1
            raw = raw[1:]

        if not raw:
            return ("rest", sustain + 1)

        value = None
        if raw.isdigit():
            value = int(raw)
        else:
            value = self.NOTE_ALIASES.get(raw.lower())
        if value is None:
            raise ValueError(f"未知音符: {symbol}")
        return ((value, octave_shift), sustain + 1)

    def value_to_freq(self, value: int, octave_shift: int = 0) -> float:
        if value not in self.NOTE_INDEX:
            raise ValueError(f"不支持的音符编号: {value}")
        semitone_offset = self.NOTE_INDEX[value]
        freq = self.base_freq * (2 ** (semitone_offset / 12))
        freq *= 2 ** octave_shift
        return freq

    def _buzz(self, freq: float, duration_s: float):
        self._ensure_board()
        on_time = max(0.05, float(duration_s))
        off_time = 0.01
        try:
            self.board.set_buzzer(int(freq), on_time, off_time, 1)
        except Exception as exc:  # noqa: BLE001
            raise RuntimeError(f"蜂鸣器指令失败: {exc}") from exc

    def play_note(self, note_value: int, duration_s: float = 0.25):
        freq = self.value_to_freq(note_value, 0)
        with self._lock:
            self._buzz(freq, duration_s)

    def play_named(self, name: str, duration_s: float = 0.25):
        value = self.note_name_to_value(name)
        if value is None:
            raise RuntimeError("未知的音符名称")
        self.play_note(value, duration_s=duration_s)

    def _parse_score(self, text: str, tempo_s: float) -> List[tuple]:
        events = []
        tokens = text.replace("\n", " ").split()
        for raw in tokens:
            try:
                parsed = self.symbol_to_value(raw)
            except ValueError as exc:
                raise RuntimeError(str(exc)) from exc
            if not parsed:
                continue
            symbol, repeat = parsed
            for _ in range(max(1, repeat)):
                if symbol == "rest":
                    events.append((None, tempo_s))
                else:
                    value, octave = symbol
                    freq = self.value_to_freq(value, octave)
                    events.append((freq, tempo_s))
        return events

    def play_score_text(self, text: str, tempo_s: float = 0.35):
        score = self._parse_score(text, tempo_s)
        if not score:
            raise RuntimeError("乐谱为空")
        if not self._lock.acquire(blocking=False):
            raise RuntimeError("有其他乐曲正在播放，请稍候")
        try:
            for freq, duration in score:
                if freq is None:
                    time.sleep(duration)
                else:
                    self._buzz(freq, duration)
                    time.sleep(max(0.01, duration * 0.2))
        finally:
            self._lock.release()

    def play_file(self, track_name: str, tempo_s: float = 0.35):
        base = os.path.join("file", "music")
        path = os.path.join(base, f"{track_name}.txt")
        if not os.path.isfile(path):
            raise RuntimeError(f"找不到乐谱文件: {track_name}.txt")
        with open(path, "r", encoding="utf-8") as fh:
            content = fh.read()
        self.play_score_text(content, tempo_s=tempo_s)


MUSIC = MusicService()
