import atexit
import inspect
import time
from dataclasses import dataclass
from typing import Optional, Tuple


@dataclass
class DriverState:
    ready: bool = False
    error: Optional[str] = None
    last_cmd: Optional[Tuple[str, int]] = None  # (dir, speed)


class RosmasterWrapper:
    def __init__(self, car_type: int = 1, devices: Optional[Tuple[str, ...]] = None):
        self.ok = False
        self.err: Optional[str] = None
        self.board = None
        self.device_used: Optional[str] = None
        devices_to_try = devices or ("/dev/myserial", "/dev/ttyUSB0", None)
        try:
            from Rosmaster_Lib import Rosmaster  # type: ignore

            init_params = set(inspect.signature(Rosmaster.__init__).parameters)
            last_err = None
            for dev in devices_to_try:
                kwargs = {}
                if "car_type" in init_params:
                    kwargs["car_type"] = car_type
                if dev:
                    if "device" in init_params:
                        kwargs["device"] = dev
                    elif "com" in init_params:
                        kwargs["com"] = dev
                    elif "port" in init_params:
                        kwargs["port"] = dev
                try:
                    self.board = Rosmaster(**kwargs)
                    self.device_used = dev or "(default)"
                    break
                except Exception as e:  # noqa: BLE001
                    last_err = str(e)
                    self.board = None
            if self.board is None:
                raise RuntimeError(last_err or "Rosmaster init failed")
            try:
                self.board.create_receive_threading()
            except Exception:
                # Some firmwares start RX internally; ignore failures here
                pass
            self.ok = True
        except Exception as e:  # noqa: BLE001
            self.err = str(e)
            self.ok = False

    def motion(self, vx=0.0, vy=0.0, vz=0.0):
        if not self.ok:
            raise RuntimeError(f"Rosmaster init failed: {self.err}")
        vx = max(-1.0, min(1.0, float(vx)))
        vy = max(-1.0, min(1.0, float(vy)))
        vz = max(-5.0, min(5.0, float(vz)))
        self.board.set_car_motion(vx, vy, vz)

    def stop(self):
        if self.ok:
            try:
                self.board.set_car_motion(0.0, 0.0, 0.0)
            except Exception:
                pass

    def cleanup(self):
        self.stop()


class DriverService:
    def __init__(self) -> None:
        self.state = DriverState()
        try:
            self._drv = RosmasterWrapper()
            self.state.ready = self._drv.ok
            self.state.error = None if self._drv.ok else self._drv.err
        except Exception as e:  # noqa: BLE001
            self._drv = None
            self.state.ready = False
            self.state.error = str(e)

        atexit.register(self.cleanup)

    def cleanup(self):
        try:
            if self._drv:
                self._drv.cleanup()
        except Exception:
            pass

    @staticmethod
    def _speed01(pct) -> float:
        try:
            s = max(0, min(100, int(pct)))
        except Exception:
            s = 70
        return round(s / 100.0, 3)

    def move(self, direction: str, speed_pct: int) -> Tuple[str, int]:
        direction = (direction or "").lower()
        sp01 = self._speed01(speed_pct)
        if not self.state.ready:
            raise RuntimeError(self.state.error or "driver not ready")

        if direction == "forward":
            # 前进：四个轮子同向前进
            self._drv.motion(vx=sp01, vy=0.0, vz=0.0)
        elif direction == "back":
            # 后退：四个轮子同向后退
            self._drv.motion(vx=-sp01, vy=0.0, vz=0.0)

        # ⭐ 左右改为原地转向（不再横移）
        elif direction == "left":
            # 左转：等价于左轮后退、右轮前进
            self._drv.motion(vx=0.0, vy=0.0, vz=sp01 * 5.0)
        elif direction == "right":
            # 右转：等价于左轮前进、右轮后退
            self._drv.motion(vx=0.0, vy=0.0, vz=-sp01 * 5.0)

        # 保留 turn_left / turn_right，与 left/right 行为一致
        elif direction == "turn_left":
            self._drv.motion(vx=0.0, vy=0.0, vz=sp01 * 5.0)
        elif direction == "turn_right":
            self._drv.motion(vx=0.0, vy=0.0, vz=-sp01 * 5.0)

        elif direction == "stop":
            self._drv.stop()
        else:
            raise ValueError("unknown direction")

        self.state.last_cmd = (direction, int(speed_pct))
        return self.state.last_cmd

    def health(self):
        return {
            "ready": self.state.ready,
            "rosmaster_ready": self.state.ready,
            "device": getattr(self._drv, "device_used", None),
            "error": self.state.error,
        }

    def beep(self, times: int = 5, duration_ms: int = 80, gap_s: float = 0.05):
        """Trigger the Rosmaster buzzer multiple times."""
        if not self.state.ready:
            raise RuntimeError(self.state.error or "driver not ready")
        board = getattr(self._drv, "board", None)
        if board is None or not hasattr(board, "set_beep"):
            raise RuntimeError("beep API not available on Rosmaster board")
        duration_ms = max(10, min(1000, int(duration_ms)))
        count = max(1, min(20, int(times)))
        gap = max(0.01, float(gap_s))
        for _ in range(count):
            try:
                board.set_beep(duration_ms)
            except Exception as exc:  # noqa: BLE001
                raise RuntimeError(f"set_beep failed: {exc}") from exc
            time.sleep(gap)

    def battery_voltage(self) -> float:
        """Read battery voltage from Rosmaster board if available."""
        if not self.state.ready:
            raise RuntimeError(self.state.error or "driver not ready")
        board = getattr(self._drv, "board", None)
        if board is None:
            raise RuntimeError("rosmaster board not initialized")
        if not hasattr(board, "get_battery_voltage"):
            raise RuntimeError("battery voltage API not available on Rosmaster board")
        try:
            volt = float(board.get_battery_voltage())
        except TypeError:
            volt = float(board.get_battery_voltage(0))
        if volt <= 0:
            raise RuntimeError("invalid voltage reading")
        return round(volt, 2)

    def set_servo(self, servo_id: int, angle: float) -> Tuple[int, float]:
        if not self.state.ready:
            raise RuntimeError(self.state.error or "driver not ready")
        angle = float(angle)
        impl = None
        if hasattr(self._drv.board, "set_bus_servo"):
            impl = getattr(self._drv.board, "set_bus_servo")
        elif hasattr(self._drv.board, "set_servo_angle"):
            impl = getattr(self._drv.board, "set_servo_angle")
        elif hasattr(self._drv.board, "bus_servo_set"):
            impl = getattr(self._drv.board, "bus_servo_set")
        if impl is None:
            raise RuntimeError("bus servo API not available on Rosmaster board")
        try:
            impl(int(servo_id), angle)
        except TypeError:
            impl(int(servo_id), int(round(angle)))
        return int(servo_id), angle


DRIVER = DriverService()
