import os
import pybullet as p
import pybullet_data
import numpy as np
import time
import math
from kuka import Kuka
from debug_controller_6d import Debug6DController

# ========== 状态机定义 ==========
STATE_FIXED = 0
STATE_MOVING_TO_POS_1 = 1
STATE_MOVING_TO_POS_2 = 2
STATE_MOVING_TO_POS_3 = 3
STATE_MOVING_TO_POS_4 = 4
# (STATE_MANUAL_JOG 已删除)

# ========== 参数定义 ==========
TIME_STEP = 1.0 / 240.0

# ========== 虚拟弹簧/阻抗参数 (手臂) ==========
SPRING_K_POS = 15.0    # N/m   
SPRING_C_POS = 3.0     # N·s/m 
SPRING_K_ROT = 5.0     # Nm/rad 
SPRING_C_ROT = 1.0     # Nm·s/rad 

# ========== (V8) 3点弹簧 ==========
# (V18) 这是机械臂末端的弹簧连接点 (相对于末端EE link)
MULTI_ATTACH_POS_EE = np.array([
    [ 0.02, 0.04, 0.21],  
    [ 0.02, -0.00, 0.21],  
    [ -0.02, 0.04, 0.23],
    [ -0.02, -0.00, 0.23]   
])
print(f"使用 {len(MULTI_ATTACH_POS_EE)} 点弹簧系统。(已下移 20cm)")

# --- (V18) 锚点位置 1 (数据) ---
ANCHOR_1A_DATA = [0.585, 0.226, 0.065] 
ANCHOR_1B_DATA = [0.585, 0.190, 0.065]
ANCHOR_1C_DATA = [0.621, 0.226, 0.065]
ANCHOR_1D_DATA = [0.621, 0.190, 0.065]
ANCHORS_DATA_1 = np.array([
    ANCHOR_1A_DATA, ANCHOR_1B_DATA, ANCHOR_1C_DATA, ANCHOR_1D_DATA
])
# (V18) 为锚点 1 指定 Mode_ID = 0 (世界坐标)
ANCHORS_DATA_1_MODE = 0

# --- (V18) 锚点位置 2 (数据) ---
ANCHOR_2A_DATA = [0.585, 0.226, 0.055] 
ANCHOR_2B_DATA = [0.585, 0.190, 0.055]
ANCHOR_2C_DATA = [0.621, 0.226, 0.055]
ANCHOR_2D_DATA = [0.621, 0.190, 0.055]
ANCHORS_DATA_2 = np.array([
    ANCHOR_2A_DATA, ANCHOR_2B_DATA, ANCHOR_2C_DATA, ANCHOR_2D_DATA
])
# (V18) 为锚点 2 指定 Mode_ID = 1 (相对 'top_id' 的局部坐标)
ANCHORS_DATA_2_MODE = 1

# --- (V18) 锚点位置 3 (数据) ---
ANCHOR_3A_DATA = [0.600, -0.005, 0.195]
ANCHOR_3B_DATA = [0.605, -0.195, 0.195]
ANCHOR_3C_DATA = [0.650, -0.005, 0.155]
ANCHOR_3D_DATA = [0.650, -0.205, 0.155]
ANCHORS_DATA_3 = np.array([
    ANCHOR_3A_DATA, ANCHOR_3B_DATA, ANCHOR_3C_DATA, ANCHOR_3D_DATA
])
# (V18) (假设) 为锚点 3 指定 Mode_ID = 1 (相对 'top_id' 的局部坐标)
ANCHORS_DATA_3_MODE = 1

# --- (V18) 锚点位置 4 (数据) ---
ANCHOR_4A_DATA = [0.58, 0.22, 0.25]
ANCHOR_4B_DATA = [0.58, 0.18, 0.25]
ANCHOR_4C_DATA = [0.62, 0.22, 0.25]
ANCHOR_4D_DATA = [0.62, 0.18, 0.25]
ANCHORS_DATA_4 = np.array([
    ANCHOR_4A_DATA, ANCHOR_4B_DATA, ANCHOR_4C_DATA, ANCHOR_4D_DATA
])
# (V18) (假设) 为锚点 4 指定 Mode_ID = 1 (相对 'top_id' 的局部坐标)
ANCHORS_DATA_4_MODE = 1


TARGET_ORN_EULER = [math.pi, 0, -1.5500] 
TARGET_ORN_QUAT = p.getQuaternionFromEuler(TARGET_ORN_EULER)

# ========== PyBullet 初始化 ==========
p.connect(p.GUI)
p.setAdditionalSearchPath(pybullet_data.getDataPath())
p.setPhysicsEngineParameter(numSolverIterations=150)
p.setTimeStep(TIME_STEP)
p.setGravity(0, 0, -9.8)

# ========== 加载场景和机器人 ==========
p.loadURDF("plane.urdf", basePosition=[0, 0, -0.05])
table_id = p.loadURDF("models/table_collision/table.urdf", [0.5, 0, -0.625], useFixedBase=True)

kuka = Kuka("models/kuka_iiwa/kuka_with_gripper2.sdf")
print("Kuka 机器人已加载。")
robot = kuka.kukaUid

# ========== 关节识别 ==========
# 1. 识别末端 link (EE)
ee_link_index = None
for i in range(p.getNumJoints(robot)):
    link_name = p.getJointInfo(robot, i)[12].decode('utf-8')
    if any(key in link_name for key in ["lbr_iiwa_link_7", "link_7", "tool0"]):
        ee_link_index = i
        break
if ee_link_index is None:
    ee_link_index = p.getNumJoints(robot) - 1
print(f"EE link index = {ee_link_index}")

# 2. (硬编码) Kuka IIWA 手臂是 7-DOF
arm_joint_indices = [0, 1, 2, 3, 4, 5, 6]
print(f"Arm joint indices = {arm_joint_indices}")

# 3. (硬编码) 夹爪的腕部旋转关节
gripper_to_arm_idx = 7
print(f"Gripper wrist joint index = {gripper_to_arm_idx}")

# 4. (硬编码) 夹爪的两个手指关节
gripper_joint_indices = [8, 11]
print(f"找到 {len(gripper_joint_indices)} 个夹爪主驱动关节: {gripper_joint_indices}")

# === 5. (必需) active DOF：所有非固定关节
active_joint_indices = [j for j in range(p.getNumJoints(robot)) if p.getJointInfo(robot, j)[2] != p.JOINT_FIXED]
arm_cols_in_active = [active_joint_indices.index(j) for j in arm_joint_indices]

# === 6. (必需) 锁定手腕旋转关节 ===
if gripper_to_arm_idx is not None:
    print(f"锁定 Gripper->Arm 关节 {gripper_to_arm_idx} 到 0 度。")
    p.setJointMotorControl2(
        robot, gripper_to_arm_idx,
        controlMode=p.POSITION_CONTROL,
        targetPosition=0.0,
        force=200.0, 
        positionGain=0.2, velocityGain=1.0
    )

kuka.reset() 
initial_arm_q = []
for j in arm_joint_indices:
    initial_arm_q.append(p.getJointState(robot, j)[0])
print(f"初始固定角度: {np.round(initial_arm_q, 2)}")

# ==============================================================================
# ========== 加载并缩放新物体 (MODIFIED SECTION) ==========
# ==============================================================================
print("在桌面上加载自定义物体...")

# 定义初始位置 (确保在机械臂工作范围内, x~0.5-0.7)
stand_initial_pos   = [0.65, 0.20, 0.05]
stand_initial_orn   = p.getQuaternionFromEuler([0, 0, 0])

body_initial_pos    = [0.65, 0.00, 0.05]
body_initial_orn    = p.getQuaternionFromEuler([0, 0, math.pi/2]) # 旋转90度

chimney_initial_pos = [0.65, -0.20, 0.05]
chimney_initial_orn = p.getQuaternionFromEuler([0, 0, 0])

top1_initial_pos    = [0.50, 0.20, 0.05]
top1_initial_orn    = p.getQuaternionFromEuler([0, 0, 0])

top2_initial_pos    = [0.50, -0.20, 0.05]
top2_initial_orn    = p.getQuaternionFromEuler([0, 0, 0])

base_initial_pos    = [0.80, 0.00, 0.00] # 放在稍微远一点的地方作为演示
base_initial_orn    = p.getQuaternionFromEuler([0, 0, 0])

# --- 1. 三角支架 (Stand) ---
try:
    stand_id = p.loadURDF("ok/stand_tri/urdf/stand_tri.urdf", 
                          basePosition=stand_initial_pos, 
                          baseOrientation=stand_initial_orn, 
                          globalScaling=2)
    print("Stand 加载成功")
except:
    print("Warning: Stand 加载失败, 使用默认替代品")
    stand_id = p.loadURDF("cube.urdf", basePosition=stand_initial_pos, globalScaling=0.5)

# --- 2. 主体 (Main Body) ---
try:
    body_id = p.loadURDF("ok/main4/urdf/main4.urdf", 
                         basePosition=body_initial_pos, 
                         baseOrientation=body_initial_orn, 
                         globalScaling=2)
    print("Main Body 加载成功")
except:
    print("Warning: Main Body 加载失败, 使用默认替代品")
    body_id = p.loadURDF("cube.urdf", basePosition=body_initial_pos, globalScaling=0.5)

# --- 3. 烟囱 (Chimney) ---
try:
    chimney_id = p.loadURDF("ok/chimney/urdf/chimney.urdf", 
                            basePosition=chimney_initial_pos, 
                            baseOrientation=chimney_initial_orn, 
                            globalScaling=2) # 也可以改成 2 如果需要变大
    print("Chimney 加载成功")
except:
    print("Warning: Chimney 加载失败, 使用蓝色方块")
    chimney_id = p.loadURDF("models/box_blue.urdf", basePosition=chimney_initial_pos, globalScaling=0.7)

# --- 4. 顶板 1 (Top 1) -> 赋值给 top_id ---
try:
    top_id = p.loadURDF("ok/top1/urdf/top1.urdf", 
                        basePosition=top1_initial_pos, 
                        baseOrientation=top1_initial_orn, 
                        globalScaling=2)
    print("Top 1 加载成功 (作为 top_id)")
except:
    print("Warning: Top 1 加载失败, 使用默认替代品")
    top_id = p.loadURDF("models/box_green.urdf", basePosition=top1_initial_pos, globalScaling=0.5)

# --- 5. 顶板 2 (Top 2) ---
try:
    top2_id = p.loadURDF("ok/top2/urdf/top2.urdf", 
                         basePosition=top2_initial_pos, 
                         baseOrientation=top2_initial_orn, 
                         globalScaling=2)
    print("Top 2 加载成功")
except:
    print("提示: Top 2 文件未找到或加载失败 (ok/top2/urdf/top2.urdf)")
    top2_id = -1

# --- 6. 基座 (Base) ---
try:
    base_id = p.loadURDF("ok/base/urdf/base.urdf", 
                         basePosition=base_initial_pos, 
                         baseOrientation=base_initial_orn,
                         useFixedBase=True, # 基座通常是固定的? 如果不是改为False
                         globalScaling=2)
    print("Base 加载成功")
except:
    print("提示: Base 文件未找到或加载失败 (ok/base/urdf/base.urdf)")
    base_id = -1

# (===== V10 障碍物列表已删除 =====)
# ===================================

# ========== 初始化状态 ==========
current_state = STATE_FIXED 

# (V18 MOD) 定义当前激活的锚点数据和其模式
active_anchors_data = ANCHORS_DATA_1
active_frame_mode = ANCHORS_DATA_1_MODE

# (V12) Kuka.py 风格的夹爪角度 (0.0=闭合, 0.4=张开)
kuka_finger_angle = 0.0 
dbg_line_ids = [-1] * len(MULTI_ATTACH_POS_EE) 

# 辅助函数
def mat_from_quat(q):
    Rflat = p.getMatrixFromQuaternion(q)
    return np.array(Rflat, dtype=float).reshape(3, 3)

# ========== (2) 新增：初始化 6D 调试控制器 ==========
# (设置一个您认为合理的初始位置，例如在桌子旁边)
initial_controller_pos = [0.5, -0.3, 0.2] 
initial_controller_orn = [0, 0, 0]
controller_6d = Debug6DController(
    initial_pos=initial_controller_pos,
    initial_orn_euler=initial_controller_orn,
    axis_length=0.1  # (设置为 0.01 即为 1cm)
)
controller_6d.print_controls() # 打印键位说明
# ===============================================


# ========== 主循环 ==========
print("\n[模式] 状态机控制 (V20 - Custom Objects)")
print("     [1]     激活弹簧 -> 位置 1 (Mode 0: World)")
print("     [2]     激活弹簧 -> 位置 2 (Mode 1: Relative)")
print("     [3]     激活弹簧 -> 位置 3 (Mode 1: Relative)")
print("     [4]     激活弹簧 -> 位置 4 (Mode 1: Relative)")
print("     [N]     夹爪 -> 张开一点")
print("     [B]     夹爪 -> 闭合一点")
print("     [R]     重置 (返回固定状态)")
print("     [ESC]   退出.")

ee_pos_world = np.array([0.0, 0.0, 0.0]) 
# (V19) Flag to check if top_pos was successfully read
top_pos_valid = False 

while True:
    
    keys_now = p.getKeyboardEvents()
    controller_6d.update(keys_now)
    
    if 27 in keys_now and keys_now[27] & p.KEY_WAS_TRIGGERED:
        break

    # (V19 MOD) Get poses for relative calculation at the top of the loop
    try:
        # Get controller's world position (assuming .pos attribute exists)
        controller_pos_world = np.array(controller_6d.pos) 
        
        # Get top_id's world pose
        # 注意: top_id 对应我们加载的 "Top 1"
        top_pos, top_orn = p.getBasePositionAndOrientation(top_id)
        top_pos = np.array(top_pos)
        
        # Calculate relative position
        R_world_from_top = mat_from_quat(top_orn)
        R_top_from_world = R_world_from_top.T # Transpose is inverse for rotation matrix
        
        vec_world = controller_pos_world - top_pos
        vec_relative = R_top_from_world @ vec_world
        
        # Create relative position string
        ctrl_rel_str = f"Ctrl_Rel(top):({vec_relative[0]:.2f}, {vec_relative[1]:.2f}, {vec_relative[2]:.2f})"
        top_pos_str = f"Top(World):({top_pos[0]:.2f}, {top_pos[1]:.2f})"
        top_pos_valid = True # (V19) Flag success
    
    except Exception as e:
        # Handle cases where top_id or controller_6d.pos might not be ready/exist
        ctrl_rel_str = "Ctrl_Rel(top): (Calc...)"
        top_pos_str = "Top: N/A"
        top_pos_valid = False
        # (V19) Define fallbacks so the spring logic doesn't crash
        top_pos = np.array([0,0,0])
        R_world_from_top = np.identity(3)

    # R: 重置
    if ord('r') in keys_now and keys_now[ord('r')] & p.KEY_WAS_TRIGGERED:
        print("\n[R] 重置机器人和物体 (返回 STATE_FIXED)...")
        kuka.reset()
        kuka_finger_angle = 0.0 
        
        initial_arm_q = []
        for j in arm_joint_indices:
            initial_arm_q.append(p.getJointState(robot, j)[0])
        current_state = STATE_FIXED 
        
        # 重置所有自定义物体
        p.resetBasePositionAndOrientation(stand_id, stand_initial_pos, stand_initial_orn)
        p.resetBasePositionAndOrientation(body_id, body_initial_pos, body_initial_orn)
        p.resetBasePositionAndOrientation(chimney_id, chimney_initial_pos, chimney_initial_orn)
        p.resetBasePositionAndOrientation(top_id, top1_initial_pos, top1_initial_orn)
        if top2_id != -1:
            p.resetBasePositionAndOrientation(top2_id, top2_initial_pos, top2_initial_orn)
        # Base 通常不需要重置如果它是 static 的，但如果是 dynamic 的则需要:
        if base_id != -1:
            p.resetBasePositionAndOrientation(base_id, base_initial_pos, base_initial_orn)

    # 状态切换
    state_changed = False
    
    # (V15 MOD) 键位 ' ' -> '1'
    if ord('1') in keys_now and keys_now[ord('1')] & p.KEY_WAS_TRIGGERED:
        if current_state == STATE_FIXED:
            # (V15 MOD)
            print("\n[1] 激活! -> STATE_MOVING_TO_POS_1")
            for j in arm_joint_indices:
                p.setJointMotorControl2(robot, j, p.VELOCITY_CONTROL, force=0)
            state_changed = True
        if current_state != STATE_MOVING_TO_POS_1:
             print("\n切换到 -> POS_1 (Mode 0: World)")
             state_changed = True
        current_state = STATE_MOVING_TO_POS_1
        # (V18 MOD) 设置数据和对应的模式
        active_anchors_data = ANCHORS_DATA_1 
        active_frame_mode = ANCHORS_DATA_1_MODE
        
    # (V15 MOD) 键位 'k' -> '2'
    if ord('2') in keys_now and keys_now[ord('2')] & p.KEY_WAS_TRIGGERED:
        if current_state == STATE_FIXED:
            # (V15 MOD)
            print("\n[2] 激活! -> STATE_MOVING_TO_POS_2")
            for j in arm_joint_indices:
                p.setJointMotorControl2(robot, j, p.VELOCITY_CONTROL, force=0)
            state_changed = True
        if current_state != STATE_MOVING_TO_POS_2:
             print("\n切换到 -> POS_2 (Mode 1: Relative)")
             state_changed = True
        current_state = STATE_MOVING_TO_POS_2
        # (V18 MOD) 设置数据和对应的模式
        active_anchors_data = ANCHORS_DATA_2 
        active_frame_mode = ANCHORS_DATA_2_MODE

    # (V15 MOD) 键位 'm' -> '3'
    if ord('3') in keys_now and keys_now[ord('3')] & p.KEY_WAS_TRIGGERED:
        if current_state == STATE_FIXED:
            # (V15 MOD)
            print("\n[3] 激活! -> STATE_MOVING_TO_POS_3")
            for j in arm_joint_indices:
                p.setJointMotorControl2(robot, j, p.VELOCITY_CONTROL, force=0)
            state_changed = True
        if current_state != STATE_MOVING_TO_POS_3:
             print("\n切换到 -> POS_3 (Mode 1: Relative)")
             state_changed = True
        current_state = STATE_MOVING_TO_POS_3
        # (V18 MOD) 设置数据和对应的模式
        active_anchors_data = ANCHORS_DATA_3 
        active_frame_mode = ANCHORS_DATA_3_MODE

    # (V15 MOD) 键位 'l' -> '4'
    if ord('4') in keys_now and keys_now[ord('4')] & p.KEY_WAS_TRIGGERED:
        if current_state == STATE_FIXED:
            # (V15 MOD)
            print("\n[4] 激活! -> STATE_MOVING_TO_POS_4")
            for j in arm_joint_indices:
                p.setJointMotorControl2(robot, j, p.VELOCITY_CONTROL, force=0)
            state_changed = True
        if current_state != STATE_MOVING_TO_POS_4:
             print("\n切换到 -> POS_4 (Mode 1: Relative)")
             state_changed = True
        current_state = STATE_MOVING_TO_POS_4
        # (V18 MOD) 设置数据和对应的模式
        active_anchors_data = ANCHORS_DATA_4 
        active_frame_mode = ANCHORS_DATA_4_MODE
            
    # (===== V12: 混合控制 - Kuka.py 风格的夹爪位置控制 =====)
    df = 0.0
    if ord('n') in keys_now and keys_now[ord('n')] & p.KEY_WAS_TRIGGERED: # 'N' 键 张开
        df = 0.5
        print("\n[N] 夹爪 -> 张开一点")
    if ord('b') in keys_now and keys_now[ord('b')] & p.KEY_WAS_TRIGGERED: # 'B' 键 闭合
        df = -0.5
        print("\n[B] 夹爪 -> 闭合一点")
    
    # (1) 更新夹爪目标角度
    kuka_finger_angle = np.clip(kuka_finger_angle + df, 0.0, 0.4)

    # (2) 应用夹爪位置控制
    p.setJointMotorControl2(robot, 8, p.POSITION_CONTROL,
                            targetPosition=-kuka_finger_angle, force=2.5)
    p.setJointMotorControl2(robot, 11, p.POSITION_CONTROL,
                            targetPosition=kuka_finger_angle, force=2.5)
    p.setJointMotorControl2(robot, 10, p.POSITION_CONTROL,
                            targetPosition=0, force=2)
    p.setJointMotorControl2(robot, 13, p.POSITION_CONTROL,
                            targetPosition=0, force=2)
    
    # =================================================================
    
    # --- 3. 控制逻辑 (状态机) ---
            
    if current_state == STATE_FIXED:
        # (===== 状态: 固定 =====)
        p.setJointMotorControlArray(
            robot, arm_joint_indices,
            controlMode=p.POSITION_CONTROL,
            targetPositions=initial_arm_q,
            forces=[240.0] * len(arm_joint_indices), 
            positionGains=[0.05] * len(arm_joint_indices), 
            velocityGains=[1.0] * len(arm_joint_indices)
        )
        if dbg_line_ids[0] != -1:
            for i in range(len(dbg_line_ids)):
                if dbg_line_ids[i] != -1:
                    p.removeUserDebugItem(dbg_line_ids[i])
                    dbg_line_ids[i] = -1
        ls_ee = p.getLinkState(robot, ee_link_index)
        ee_pos_world = np.array(ls_ee[0])
    
    else:
        # (===== 状态: 弹簧控制 (POS_1, 2, 3, 4) =====)
        
        # 1) 收集状态
        q_active  = []
        qd_active = []
        for j in active_joint_indices:
            js = p.getJointState(robot, j)
            q_active.append(js[0])
            qd_active.append(js[1])
        q_active  = np.array(q_active, dtype=float)
        qd_active = np.array(qd_active, dtype=float)
        zeros_jac = [0.0] * len(active_joint_indices)
        
        # 2) 重力补偿
        qdd_active = [0.0] * len(active_joint_indices)
        tau_g_active = np.array(p.calculateInverseDynamics(
            robot, list(q_active), [0.0]*len(active_joint_indices), qdd_active
        ))
        tau_g_arm = tau_g_active[arm_cols_in_active]

        # 3) 获取末端状态
        ls_ee = p.getLinkState(robot, ee_link_index, computeLinkVelocity=1, computeForwardKinematics=1)
        ee_pos_world = np.array(ls_ee[0])
        ee_orn_world = np.array(ls_ee[1])
        ee_vel_world = np.array(ls_ee[6])
        ee_ang_vel_world = np.array(ls_ee[7])
        R_world_from_ee = mat_from_quat(ee_orn_world)

        # --- 4a. (位置) tau_pos ---
        tau_pos_spring_arm = np.zeros(7)
        F_mag_total = 0.0

        # (V18 MOD) 根据当前激活的 active_frame_mode 计算有效的锚点世界坐标
        effective_anchors = np.zeros_like(active_anchors_data)
        if active_frame_mode == 1:
            # 模式 1: active_anchors_data (数据) 是相对于 top_id 的
            if top_pos_valid:
                for i_anchor in range(len(active_anchors_data)):
                    effective_anchors[i_anchor] = top_pos + R_world_from_top @ active_anchors_data[i_anchor]
            else:
                # Fallback if top_id failed (e.g., first frame)
                effective_anchors = active_anchors_data
        else:
            # 模式 0: active_anchors_data (数据) 已经是世界坐标
            effective_anchors = active_anchors_data
        
        for i in range(len(MULTI_ATTACH_POS_EE)): 
            attach_pos_local = MULTI_ATTACH_POS_EE[i]
            # (V15 MOD) 使用 effective_anchors
            anchor_pos_world = effective_anchors[i] 
            
            r_world = R_world_from_ee @ attach_pos_local
            current_attach_pos_world = ee_pos_world + r_world
            current_attach_vel_world = ee_vel_world + np.cross(ee_ang_vel_world, r_world)
            
            disp = current_attach_pos_world - anchor_pos_world
            vel = current_attach_vel_world
            F_i = -SPRING_K_POS * disp - SPRING_C_POS * vel
            F_mag_total += float(np.linalg.norm(F_i)) 

            Jt_i, Jr_i = p.calculateJacobian(
                robot, ee_link_index, list(attach_pos_local),
                list(q_active), list(zeros_jac), list(zeros_jac)
            )
            Jv_i_arm = np.array(Jt_i)[:, arm_cols_in_active]
            tau_i = Jv_i_arm.T @ F_i
            tau_pos_spring_arm += tau_i
            
            # (V14) 根据4个状态设置不同颜色
            if current_state == STATE_MOVING_TO_POS_1:
                color = [0, 1, 0] # Green
            elif current_state == STATE_MOVING_TO_POS_2:
                color = [0, 0, 1] # Blue
            elif current_state == STATE_MOVING_TO_POS_3:
                color = [1, 0, 1] # Magenta
            else: # POS_4
                color = [1, 1, 0] # Yellow
            
            dbg_line_ids[i] = p.addUserDebugLine(
                current_attach_pos_world, anchor_pos_world, color, 2.0, 0,
                replaceItemUniqueId=dbg_line_ids[i]
            )

        # --- 5) 合成总扭矩 ---
        tau_cmd_arm = tau_g_arm + tau_pos_spring_arm
        
        # --- 6) 下发扭矩 ---
        p.setJointMotorControlArray(
            robot, arm_joint_indices,
            controlMode=p.TORQUE_CONTROL,
            forces=list(tau_cmd_arm)
        )

    # --- 7. 状态输出 ---
    pose_str = f"Pose:({ee_pos_world[0]:.2f}, {ee_pos_world[1]:.2f}, {ee_pos_world[2]:.2f})"
    gripper_state_str = f"Angle:{kuka_finger_angle:.2f}"
    
    # (V19) ctrl_str (世界坐标) 和 ctrl_rel_str (相对坐标) 都在循环顶部计算好了
    # ctrl_str = controller_6d.get_pose_str()
    
    frame_mode_str = "REL" if active_frame_mode == 1 else "WORLD"
    
    if current_state == STATE_FIXED:
        state_str = "STATE: FIXED (LOCKED)"
        # 为了不刷屏太快，稍微sleep久一点点或者不打印
        # print(f"\r[{state_str}] | {top_pos_str} | {ctrl_rel_str} | Gripper:{gripper_state_str} | {pose_str}", end=" ")
    
    else:
        # (V14) 弹簧模式的状态输出
        if current_state == STATE_MOVING_TO_POS_1:
            state_str = "POS_1"
            color_str = "\033[92m" # Green
        elif current_state == STATE_MOVING_TO_POS_2:
            state_str = "POS_2"
            color_str = "\033[94m" # Blue
        elif current_state == STATE_MOVING_TO_POS_3:
            state_str = "POS_3"
            color_str = "\033[95m" # Magenta
        else: # POS_4
            state_str = "POS_4"
            color_str = "\033[93m" # Yellow
        
        reset_color = "\033[0m"
        print(f"\r[{color_str}SPRING{reset_color} -> {state_str}] | Frame: {frame_mode_str} | |F|={F_mag_total:4.1f} | {top_pos_str} | {ctrl_rel_str} | {pose_str}", end="")

    # --- 8. 仿真步进 ---
    p.stepSimulation()

    time.sleep(TIME_STEP)

print("\n仿真结束。")
controller_6d.remove()
p.disconnect()