# run_base4.py

import time
import numpy as np
import pybullet as p

# 从 kuka.py 导入 KukaCamEnv4 环境类
# 这个类负责创建PyBullet仿真世界、加载机器人和物体、执行动作等
from kuka import KukaCamEnv4

# 从 agent.py 导入 base4 控制器函数
# 这是我们的“专家大脑”，一个基于规则的策略，用于决定机器人每一步的动作
# 我们也需要导入其依赖的函数
from agent import base_template, xdist, ydist, da

# ==============================================================================
# 为了方便理解和注释，我们将 agent.py 中的相关控制器函数复制到这里。
# 在实际项目中，可以直接从 agent.py 导入。
# ==============================================================================

def base4(s):
    """
    base4 控制器，专门用于任务4（堆叠两个1x1小物块）。
    它调用一个通用的模板函数 base_template，并传入针对任务4的特定高度参数。
    """
    # z1: 机械臂在抓取第一个物块前，移动到物块上方的安全悬停高度的偏移量。   我觉得可能是重心偏移距离
    #     目标高度 = 物块高度 + 0.03 + 0.24。
    # z2: (在此策略中未使用) 通常用于抓取后提升到的高度。
    # z3: 机械臂抓取物块后，在移动到目标位置前，需要提升到的一个安全高度。
    #     这里是 0.1，表示先将抓到的物块提升到 0.1 的高度再平移。
    return base_template(s, z1=0.2, z2=0.35, z3=0.1)


def commented_base_template(s, z1, z2, z3, K=5):
    """
    这是一个带有详细注释的 base_template 版本，用于解释其内部逻辑。
    它根据当前状态 s 来决定下一步的动作。

    参数 s (state) 是一个包含20个元素的状态向量，其内容大致如下：
    s[0:3]:   机械臂末端执行器（夹爪）的位置 (x, y, z)
    s[3:6]:   机械臂末端的姿态 (roll, pitch, yaw)
    s[6]:     夹爪手指的开合角度
    s[7]:     夹爪感受到的力（用于判断是否抓住了物体）
    s[8:11]:  第一个物体（绿色方块）的位置 (x, y, z)
    s[11:14]: 第一个物体的姿态 (roll, pitch, yaw)
    s[14:17]: 第二个物体（紫色方块，在任务4中是另一个同类方块）的位置 (x, y, z)
    s[17:20]: 第二个物体的姿态 (roll, pitch, yaw)
    """
    # 从状态向量 s 中解析出需要的信息
    x_n, y_n, height = s[0], s[1], s[2]          # 机械臂末端的位置
    gripper_angle = s[5]                         # 机械臂末端的朝向 (yaw)
    finger_angle = s[6]                          # 夹爪当前开合角度
    finger_force = s[7]                          # 夹爪受力

    pos1, orn1 = s[8:11], s[11:14]               # 第一个物体的位姿
    pos2, orn2 = s[14:17], s[17:20]               # 第二个物体的位姿

    # 初始化一个5维的动作向量 [dx, dy, dz, da, df]
    # 分别对应 x,y,z位移，yaw角变化，夹爪开合变化
    action = np.zeros(5)

    # *** 核心判断逻辑 1: 判断夹爪是否为空 ***
    # 通过检查夹爪受力 finger_force 是否小于1来判断。
    # 如果小于1，意味着没抓住东西，当前目标是去抓取第一个物体 (pos1)。


        #action[0] -> dx: 沿 X 轴方向的位移变化量。
        #action[1] -> dy: 沿 Y 轴方向的位移变化量。
        #action[2] -> dz: 沿 Z 轴方向的位移变化量。
        #action[3] -> da: 绕 Z 轴的旋转角度变化量。
        #action[4] -> df: 夹爪手指开合角度的变化量。
    
    if finger_force < 1:
        # --- 阶段一：抓取第一个物体 ---

        # action[0, 1]: 控制机械臂在XY平面上移动，对准第一个物体
        action[0] = np.tanh(xdist(pos1, orn1, x_n) * K)
        action[1] = np.tanh(ydist(pos1, orn1, y_n) * K)
        action[4] = da(0.4, finger_angle)    #张开

        # action[2]: 控制机械臂在Z轴上移动。
        # 目标高度 = 物体1的高度(pos1[2]) + 一个小的固定偏移(0.03) + 预设的悬停高度偏移(z1)
        # (目标高度 - 当前高度) 的差值经过tanh函数处理，变成一个平滑的速度指令

        # 原本的代码
        # action[2] = np.tanh(K * (pos1[2] + 0.002 + z1 - height))

        # 修改版，带打印
        val = K * (pos1[2] + 0.002 + z1 - height)
        print(f"Debug: pos1[2]={pos1[2]}, z1={z1}, height={height}, K={K}, raw={val}, tanh={np.tanh(val)}")
        action[2] = np.tanh(val)


        # action[3]: 控制机械臂末端的旋转，使其朝向与物体1的朝向(orn1[2])一致
        action[3] = da(gripper_angle, orn1[2])

        # action[4]: 控制夹爪的开合。da(0, finger_angle) 表示目标是闭合夹爪（目标角度为0）
        # 这里有一个条件判断，当机械臂高度低于设定的悬停高度时，开始闭合夹爪。
        if height <  pos1[2] + 0.02 + z1:
           
            action[4] = da(0, finger_angle)
       

    # *** 核心判断逻辑 2: 如果夹爪抓住了物体 ***
    # finger_force >= 1，说明已经成功抓取第一个物体。
    # 当前目标是将其移动到第二个物体 (pos2) 的正上方并放下。
    else:
        # --- 阶段二：移动并放置物体 ---

        # action[0, 1]: 控制机械臂在XY平面上移动。
        # 目标位置是第二个物体(pos2)的XY坐标。
        # (pos2[0] - pos1[0]) 计算的是抓在手上的物体(pos1)与目标位置(pos2)的XY误差。
        action[0] = np.tanh((pos2[0] - pos1[0]) * K)
        action[1] = np.tanh((pos2[1] - pos1[1]) * K)

        # action[2]: 控制Z轴。首先，将抓取的物体提升到一个安全高度 z3，以避免在平移过程中碰到障碍。
        action[2] = np.tanh(K * (z3 - pos1[2] + 0.03))

        # action[3]: 控制旋转，使抓取的物体姿态与目标位置(第二个物体)的姿态对齐。
        action[3] = da(gripper_angle, orn2[2])

        # action[4]: 保持夹爪闭合，因为还在运输过程中。
        action[4] = da(0, finger_angle)


        # *** 核心判断逻辑 3: 判断是否到达目标正上方 (距离判断) ***
        # 计算抓取的物体(pos1)与目标位置(pos2)在XY平面上的欧式距离。
        # 如果距离小于0.01米 (1厘米)，则认为已经精确对准，可以开始执行下降和放置动作。
        dist_xy = np.sqrt((pos2[0] - pos1[0]) ** 2 + (pos2[1] - pos1[1]) ** 2)
        if dist_xy < 0.01:
            # --- 阶段三：垂直下降并释放 ---

            # 到达正上方后，覆盖之前的Z轴动作，开始垂直下降。
            # 新的目标高度是一个固定的值 0.15 (这个值是试出来的)。
            action[2] = np.tanh(5 * (0.15 - pos1[2]))

            # 当物体高度下降到目标高度 0.15 以下时，开始释放物体。
            # da(0.2, finger_angle) 表示让夹爪张开到一个中间角度0.2，而不是完全张开。
            if pos1[2] < 0.15:
                action[4] = da(0.2, finger_angle)
    return action


if __name__ == '__main__':
    # 1. 准备环境
    print("正在初始化PyBullet环境...")
    # 实例化环境，renders=True 会显示图形化界面
    # image_output=False 因为我们只用状态向量，不需要摄像头图像
    env = KukaCamEnv4(renders=True, image_output=False)

    # 设定模拟的总轮数 (episodes)
    num_episodes = 10

    # 2. 编写主循环
    for i in range(num_episodes):
        print(f"\n--- 开始第 {i+1}/{num_episodes} 轮模拟 ---")

        # a. 重置环境，获取初始状态
        # s 是包含了机器人和物体位姿等信息的20维向量
        _, s = env.reset()

        # 每轮任务最多执行的步数，防止无限循环
        max_steps = 200
        total_reward = 0

        # b. 循环执行每一步
        for step in range(max_steps):
            # 1. 计算动作：将当前状态 s 传入我们带注释的控制器函数
            # action 是一个包含5个值的numpy数组，表示机器人下一步的动作
            # 这里我们使用带注释的函数版本，以便理解
            action = commented_base_template(s, z1=0.2, z2=0.35, z3=0.1, K=5)

            # 2. 执行动作：将动作应用到环境中
            # 环境会返回执行后的新状态、奖励和任务是否结束的标志
            o_next, s_next, reward, done = env.step(action)
            
            # 3. 更新状态
            s = s_next
            total_reward += reward

            # 稍微暂停一下，让画面看起来更连贯 (240Hz是pybullet默认频率)
            time.sleep(1./15.)

            # 4. 判断结束
            if done:
                if total_reward > 0:
                    print(f"第 {i+1} 轮成功！在第 {step+1} 步完成任务。")
                else:
                    print(f"第 {i+1} 轮失败，可能是物体掉落或超时。")
                break
        
        if not done:
            print(f"第 {i+1} 轮超时，在 {max_steps} 步后仍未完成。")
            
    print("\n所有模拟轮数已完成。")
    # PyBullet环境会在程序结束时自动断开连接