#!/usr/bin/env python3
# -*- coding: utf-8 -*-

import argparse
import time
import os
import pybullet as p
import pybullet_data
from math import pi

def draw_axes(body_id, link_id, length=0.05, life_time=0):
    """在指定link处画坐标轴（仅GUI下可见）"""
    pos, orn = (p.getBasePositionAndOrientation(body_id) if link_id == -1
                else p.getLinkState(body_id, link_id)[:2])
    rot = p.getMatrixFromQuaternion(orn)

    x_dir = [rot[0], rot[3], rot[6]]
    y_dir = [rot[1], rot[4], rot[7]]
    z_dir = [rot[2], rot[5], rot[8]]
    def add_line(dir_vec, rgb):
        to_pt = [pos[i] + dir_vec[i]*length for i in range(3)]
        p.addUserDebugLine(pos, to_pt, rgb, lineWidth=3, lifeTime=life_time)
    add_line(x_dir, [1,0,0])   
    add_line(y_dir, [0,1,0])  
    add_line(z_dir, [0,0,1])   

def world_com_of_link(body_id, link_id):
    dyn = p.getDynamicsInfo(body_id, link_id)
    mass, _, _, local_inertial_pos, local_inertial_orn = dyn[0], dyn[1], dyn[2], dyn[3], dyn[4]
    if link_id == -1:
        base_pos, base_orn = p.getBasePositionAndOrientation(body_id)
        com_pos, _ = p.multiplyTransforms(base_pos, base_orn, local_inertial_pos, local_inertial_orn)
    else:
        link_state = p.getLinkState(body_id, link_id, computeForwardKinematics=True)
        link_world_pos, link_world_orn = link_state[0], link_state[1]
        com_pos, _ = p.multiplyTransforms(link_world_pos, link_world_orn, local_inertial_pos, local_inertial_orn)
    return com_pos, mass

def main():
    parser = argparse.ArgumentParser()
    parser.add_argument("--gui", action="store_true", help="使用 GUI 显示并绘制坐标轴/标记")
    parser.add_argument("--urdf", default="models/lego/model_plate_1x1.urdf", help="URDF 路径（相对或绝对）")
    parser.add_argument("--gravity", type=float, default=0.0, help="重力（默认 0，纯几何检查）")
    args = parser.parse_args()

    cid = p.connect(p.GUI if args.gui else p.DIRECT)
    p.setAdditionalSearchPath(pybullet_data.getDataPath())
    p.setGravity(0, 0, args.gravity)

    urdf_path = args.urdf
    if not os.path.isabs(urdf_path):
        urdf_path = os.path.abspath(urdf_path)
    if not os.path.exists(urdf_path):
        raise FileNotFoundError(f"URDF 未找到：{urdf_path}")

    flags = p.URDF_MAINTAIN_LINK_ORDER | p.URDF_USE_INERTIA_FROM_FILE
    body = p.loadURDF(urdf_path, basePosition=[0,0,0], baseOrientation=[0,0,0,1], useFixedBase=True, flags=flags)

    base_pos, base_orn = p.getBasePositionAndOrientation(body)
    aabb_min, aabb_max = p.getAABB(body, -1)
    height = aabb_max[2] - aabb_min[2]
    com_world, mass = world_com_of_link(body, -1)

    print("=== Base Link（-1）===")
    print(f"Base 世界位置: {base_pos}")
    print(f"Base 世界姿态(四元数): {base_orn}")
    print(f"Base AABB min: {aabb_min}, max: {aabb_max}")
    print(f"Base 高度: {height:.6f} m")
    print(f"Base 质心位置 COM: {com_world}, 质量: {mass:.6f} kg")

    num_joints = p.getNumJoints(body)
    print(f"\nLink 数量（不含 base）: {num_joints}")
    for link_id in range(num_joints):
        info = p.getJointInfo(body, link_id)
        link_name = info[12].decode("utf-8") if isinstance(info[12], bytes) else info[12]
        aabb_min_l, aabb_max_l = p.getAABB(body, link_id)
        h_l = aabb_max_l[2] - aabb_min_l[2]
        com_l, m_l = world_com_of_link(body, link_id)
        print(f"\n--- Link {link_id}: {link_name} ---")
        print(f"AABB min: {aabb_min_l}, max: {aabb_max_l}")
        print(f"高度: {h_l:.6f} m")
        print(f"COM: {com_l}, 质量: {m_l:.6f} kg")

    if args.gui:
        # 画世界坐标轴
        p.addUserDebugLine([0,0,0], [0.1,0,0], [1,0,0], 3, 0)
        p.addUserDebugLine([0,0,0], [0,0.1,0], [0,1,0], 3, 0)
        p.addUserDebugLine([0,0,0], [0,0,0.1], [0,0,1], 3, 0)

        # 画 base link 的坐标轴与质心
        draw_axes(body, -1, length=0.05, life_time=0)
        # 质心小十字
        com = com_world
        s = 0.01
        p.addUserDebugLine([com[0]-s, com[1], com[2]], [com[0]+s, com[1], com[2]], [1,1,0], 2, 0)
        p.addUserDebugLine([com[0], com[1]-s, com[2]], [com[0], com[1]+s, com[2]], [1,1,0], 2, 0)
        p.addUserDebugText("COM", [com[0], com[1], com[2]+0.01], [1,1,0], 1.5, 0)

        print("\nGUI 模式已开启，按 Ctrl+C 退出。")
        try:
            while True:
                p.stepSimulation()
                time.sleep(1.0/120.0)
        except KeyboardInterrupt:
            pass

    p.disconnect()

if __name__ == "__main__":
    main()
