import pybullet as p
import pybullet_data
import numpy as np
import time
import math
import tkinter as tk
from tkinter import ttk
import threading

# 从您提供的 kuka.py 文件中引入 Kuka 类
# 确保 kuka.py 和 models 文件夹与此脚本在同一目录下
from kuka import Kuka

# ========== 1. GUI 控制面板类 (已修改) ==========
class ControlPanel:
    def __init__(self, shared_state):
        self.root = tk.Tk()
        self.root.title("Kuka Control Panel")
        self.root.geometry("300x250") # 窗口大小已调整
        self.shared_state = shared_state

        self.entries = {}
        self.create_widgets()
        
        # 优雅地关闭线程
        self.root.protocol("WM_DELETE_WINDOW", self.on_closing)

    def create_widgets(self):
        frame = ttk.Frame(self.root, padding="10")
        frame.grid(row=0, column=0, sticky=(tk.W, tk.E, tk.N, tk.S))

        # --- 位置输入 ---
        pos_label = ttk.Label(frame, text="Target Position (m)", font=("Arial", 10, "bold"))
        pos_label.grid(row=0, column=0, columnspan=2, pady=5)
        
        labels_pos = ["Target X", "Target Y", "Target Z"]
        keys_pos = ["x", "y", "z"]
        for i, label_text in enumerate(labels_pos):
            label = ttk.Label(frame, text=label_text)
            label.grid(row=i + 1, column=0, sticky=tk.W, padx=5, pady=2)
            entry = ttk.Entry(frame, width=15)
            entry.grid(row=i + 1, column=1, padx=5, pady=2)
            self.entries[keys_pos[i]] = entry

        # --- 姿态输入部分已被移除 ---

        # --- 按钮 ---
        button_frame = ttk.Frame(frame)
        # 按钮的行号已调整
        button_frame.grid(row=4, column=0, columnspan=2, pady=20)

        go_button = ttk.Button(button_frame, text="Go to Target", command=self.go_to_target)
        go_button.pack(side=tk.LEFT, padx=5)

        get_button = ttk.Button(button_frame, text="Get Current Pose", command=self.get_current_pose)
        get_button.pack(side=tk.LEFT, padx=5)
        
    def go_to_target(self):
        try:
            # 只更新位置，姿态由键盘控制
            pos = [float(self.entries["x"].get()), float(self.entries["y"].get()), float(self.entries["z"].get())]
            
            with self.shared_state['lock']:
                self.shared_state['target_pos'] = pos
                self.shared_state['go_button_pressed'] = True
            print("GUI指令: 移动到目标位置。")
        except ValueError:
            print("错误: 请在所有输入框中输入有效的数字。")

    def get_current_pose(self):
        with self.shared_state['lock']:
            pos = self.shared_state['current_pos']
        
        if pos:
            # 只更新位置输入框
            keys = ["x", "y", "z"]
            for key, value in zip(keys, pos):
                self.entries[key].delete(0, tk.END)
                self.entries[key].insert(0, f"{value:.4f}")
            print("GUI指令: 已将当前位置同步到GUI。")

    def run(self):
        self.get_current_pose() # 初始化时获取一次姿态
        self.root.mainloop()

    def on_closing(self):
        with self.shared_state['lock']:
            self.shared_state['gui_running'] = False
        self.root.destroy()
        print("GUI窗口已关闭。")

# ========== 参数定义 ==========
TIME_STEP = 1. / 240.
MOVE_STEP = 0.0002
ANGLE_STEP = 0.002
FINGER_STEP = 0.0001

# ========== PyBullet 初始化 ==========
p.connect(p.GUI)
p.setAdditionalSearchPath(pybullet_data.getDataPath())
p.setPhysicsEngineParameter(numSolverIterations=150)
p.setTimeStep(TIME_STEP)
p.setGravity(0, 0, -9.8)

# ========== 加载场景和机器人 ==========
p.loadURDF("plane.urdf", basePosition=[0, 0, -0.05])
p.loadURDF("models/table_collision/table.urdf", [0.5, 0, -0.625], useFixedBase=True)
kuka = Kuka("models/kuka_iiwa/kuka_with_gripper2.sdf")
print("Kuka 机器人已加载。")

# ========== 加载并缩放新物体 (已修改以支持重置) ==========
print("在原点加载两个缩放后的物体...")

# <--- 新增: 定义物体的初始姿态，方便重置 ---
stand_initial_pos = [0.6, 0.2, 0.1]
stand_initial_orn = p.getQuaternionFromEuler([0, 0, 0])

body_initial_pos = [0.6, 0, 0.2]
body_initial_orn = p.getQuaternionFromEuler([0, 0, 0])

top_initial_pos = [0.6, 0.2, 0.2]
top_initial_orn = p.getQuaternionFromEuler([math.pi/2, 0, 0])

# <--- 修改: 存储加载物体的ID，以便后续操作 ---
stand_id = p.loadURDF("models/house/stand.urdf", basePosition=stand_initial_pos, baseOrientation=stand_initial_orn, globalScaling=0.5)
body_id = p.loadURDF("models/house/main_body_fixed.urdf", basePosition=body_initial_pos, baseOrientation=body_initial_orn, globalScaling=0.5)
top_id = p.loadURDF("models/house/top.urdf", basePosition=top_initial_pos, baseOrientation=top_initial_orn, globalScaling=0.5)

# ========== 初始化控制状态和线程共享数据 ==========
initial_observation = kuka.getObservation()
target_pos = list(initial_observation[0:3])
target_euler = list(initial_observation[3:6])
target_euler[2] = np.pi
finger_angle = initial_observation[6]

# ========== 定义一个可以通过'm'键触发的自定义初始姿态 ========== # <--- 新增
# 在这里定义您想要的 s[0] 到 s[6] 的值
custom_target_pos = [0.5856, 0.2032, 0.50]  # s[0], s[1], s[2] (位置 x, y, z) mmmm
custom_target_euler = [math.pi, 0, -1.5500]  # s[3], s[4], s[5] (姿态 roll, pitch, yaw)
custom_finger_angle = 0.2             # s[6] (手指开合角度, 0.4为最开)

custom_target_pos1 = [0.5856, 0.2032, 0.3830]  # s[0], s[1], s[2] (位置 x, y, z) cccc
custom_target_euler1 = [math.pi, 0, -1.5500]  # s[3], s[4], s[5] (姿态 roll, pitch, yaw)
custom_finger_angle1 = 0.2             # s[6] (手指开合角度, 0.4为最开)


custom_target_pos2 = [0.5856, 0.2032, 0.320]  # s[0], s[1], s[2] (位置 x, y, z) bbbb
custom_target_euler2 = [math.pi, 0, -1.5500]  # s[3], s[4], s[5] (姿态 roll, pitch, yaw)
custom_finger_angle2 = 0            # s[6] (手指开合角度, 0.4为最开)








#安全高度+共线

custom_target_pos3 = [0.5856, -0.02838, 0.50] # s[0], s[1], s[2] (位置 x, y, z) tttt
custom_target_euler3 = [math.pi, 0.7646, -1.5900]  # s[3], s[4], s[5] (姿态 roll, pitch, yaw)
custom_finger_angle3 = 0            # s[6] (手指开合角度, 0.4为最开)




#0是最终的目标向量
custom_target_pos0 = [0.5856, -0.1415, 0.3488]  # s[0], s[1], s[2] (位置 x, y, z) yyyy
custom_target_euler0 = [math.pi, 0.7646, -1.5900]  # s[3], s[4], s[5] (姿态 roll, pitch, yaw)
custom_finger_angle0 = 0            # s[6] (手指开合角度, 0.4为最开)

custom_target_pos4 = [0.5856, -0.1415, 0.3888]  # s[0], s[1], s[2] (位置 x, y, z) yyyy
custom_target_euler4 = [math.pi, 0.7646, -1.5900]  # s[3], s[4], s[5] (姿态 roll, pitch, yaw)
custom_finger_angle4 = 0            # s[6] (手指开合角度, 0.4为最开)
# ================================================================= #


#s[0]: 0.5856 (x) | s[1]: 0.2032 (y) | s[2]: 0.3302 (z) | s[3]: 3.1416 (R) | s[4]: 0.0000 (P) | s[5]: -1.5500 (Y) | s[6]: -0.0000 (f) 
#insite
#s[0]: 0.5850 (x) | s[1]: -0.1415 (y) | s[2]: 0.3488 (z) | s[3]: -3.1416 (R) | s[4]: -0.7646 (P) | s[5]: 1.5916 (Y) | s[6]: -0.0000 (f)   

shared_state = {
    'target_pos': target_pos[:],
    'target_euler': target_euler[:],
    'current_pos': target_pos[:],
    'current_euler': target_euler[:],
    'go_button_pressed': False,
    'gui_running': True,
    'lock': threading.Lock()
}

# ========== 启动GUI线程 ==========
gui = ControlPanel(shared_state)
gui_thread = threading.Thread(target=gui.run, daemon=True)
gui_thread.start()
print("GUI控制面板已在独立线程中启动。")

# ========== 打印控制说明 ==========
print("-" * 60)
print("Kuka 机械臂高级键盘控制说明:")
print("  --- 位置控制 (Position) ---")
print("  - X 轴: ← (Left) / → (Right)")
print("  - Y 轴: ↓ (Down) / ↑ (Up)")
print("  - Z 轴: X (下降) / Z (上升)")
print("\n  --- 姿态控制 (Orientation) ---")
print("  - 翻滚 (Roll):  A / D")
print("  - 俯仰 (Pitch): I / K")
print("  - 偏航 (Yaw):   J / L")
print("\n  --- 夹爪控制 (Gripper) ---")
print("  - 张开/闭合: G / H")
print("\n  --- 其他 ---")
print("  - 重置为默认姿态: R") # <--- 修改
print("  - 移动到自定义姿态: M") # <--- 新增
print("  - 关闭仿真或GUI窗口即可退出程序")
print("-" * 60)


# ========== 主循环 ==========
while True:
    # 检查GUI是否仍在运行
    with shared_state['lock']:
        if not shared_state['gui_running']:
            break

    # --- 1. 处理来自GUI的指令 ---
    with shared_state['lock']:
        if shared_state['go_button_pressed']:
            target_pos = shared_state['target_pos']
            shared_state['go_button_pressed'] = False # 重置标志位

    # --- 2. 处理键盘输入 ---
    keys = p.getKeyboardEvents()
    
    if p.B3G_LEFT_ARROW in keys and keys[p.B3G_LEFT_ARROW] & p.KEY_IS_DOWN: target_pos[0] -= MOVE_STEP
    if p.B3G_RIGHT_ARROW in keys and keys[p.B3G_RIGHT_ARROW] & p.KEY_IS_DOWN: target_pos[0] += MOVE_STEP
    if p.B3G_DOWN_ARROW in keys and keys[p.B3G_DOWN_ARROW] & p.KEY_IS_DOWN: target_pos[1] -= MOVE_STEP
    if p.B3G_UP_ARROW in keys and keys[p.B3G_UP_ARROW] & p.KEY_IS_DOWN: target_pos[1] += MOVE_STEP
    if ord('x') in keys and keys[ord('x')] & p.KEY_IS_DOWN: target_pos[2] -= MOVE_STEP # Z轴下降
    if ord('z') in keys and keys[ord('z')] & p.KEY_IS_DOWN: target_pos[2] += MOVE_STEP # Z轴上升

    if ord('a') in keys and keys[ord('a')] & p.KEY_IS_DOWN: target_euler[0] -= ANGLE_STEP
    if ord('d') in keys and keys[ord('d')] & p.KEY_IS_DOWN: target_euler[0] += ANGLE_STEP
    if ord('i') in keys and keys[ord('i')] & p.KEY_IS_DOWN: target_euler[1] -= ANGLE_STEP
    if ord('k') in keys and keys[ord('k')] & p.KEY_IS_DOWN: target_euler[1] += ANGLE_STEP
    if ord('j') in keys and keys[ord('j')] & p.KEY_IS_DOWN: target_euler[2] -= ANGLE_STEP
    if ord('l') in keys and keys[ord('l')] & p.KEY_IS_DOWN: target_euler[2] += ANGLE_STEP
    
    if ord('g') in keys and keys[ord('g')] & p.KEY_IS_DOWN: finger_angle += FINGER_STEP
    if ord('h') in keys and keys[ord('h')] & p.KEY_IS_DOWN: finger_angle -= FINGER_STEP
    
    # 'R' 键用于重置到默认姿态
    if ord('r') in keys and keys[ord('r')] & p.KEY_WAS_TRIGGERED:
        print("\n重置机器人到默认姿态 (R)...")
        kuka.reset()
        initial_observation = kuka.getObservation()
        target_pos = list(initial_observation[0:3])
        target_euler = list(initial_observation[3:6]); target_euler[2] = np.pi
        finger_angle = initial_observation[6]
        # 2. 重置场景中的其他物体到其加载时的姿态 <--- 新增功能
        p.resetBasePositionAndOrientation(stand_id, stand_initial_pos, stand_initial_orn)
        p.resetBasePositionAndOrientation(body_id, body_initial_pos, body_initial_orn)
        p.resetBasePositionAndOrientation(top_id, top_initial_pos, top_initial_orn)


    # 'M' 键用于移动到自定义姿态 # <--- 新增
    if ord('m') in keys and keys[ord('m')] & p.KEY_WAS_TRIGGERED:
        print("\n移动到自定义姿态 (M)...")
        target_pos = custom_target_pos[:] # 使用切片创建副本, 防止后续操作修改原始值
        target_euler = custom_target_euler[:]
        finger_angle = custom_finger_angle

    
    # 'c' 键用于移动到自定义姿态 # <--- 新增
    if ord('c') in keys and keys[ord('c')] & p.KEY_WAS_TRIGGERED:
        print("\n移动到自定义姿态 (c)...")
        target_pos = custom_target_pos1[:] # 使用切片创建副本, 防止后续操作修改原始值
        target_euler = custom_target_euler1[:]
        finger_angle = custom_finger_angle1

     
    # 'b' 键用于移动到自定义姿态 # <--- 新增
    if ord('b') in keys and keys[ord('b')] & p.KEY_WAS_TRIGGERED:
        print("\n移动到自定义姿态 (b)...")
        target_pos = custom_target_pos2[:] # 使用切片创建副本, 防止后续操作修改原始值
        target_euler = custom_target_euler2[:]
        finger_angle = custom_finger_angle2

    # 't' 键用于移动到自定义姿态 # <--- 新增
    if ord('t') in keys and keys[ord('t')] & p.KEY_WAS_TRIGGERED:
        print("\n移动到自定义姿态 (t)...")
        target_pos = custom_target_pos3[:] # 使用切片创建副本, 防止后续操作修改原始值
        target_euler = custom_target_euler3[:]
        finger_angle = custom_finger_angle3

    # 'y' 键用于移动到自定义姿态 # <--- 新增
    if ord('y') in keys and keys[ord('y')] & p.KEY_WAS_TRIGGERED:
        print("\n移动到自定义姿态 (y)...")
        target_pos = custom_target_pos4[:] # 使用切片创建副本, 防止后续操作修改原始值
        target_euler = custom_target_euler4[:]
        finger_angle = custom_finger_angle4

#s[0]: 0.5850 (x) | s[1]: -0.1415 (y) | s[2]: 0.3488 (z) | s[3]: -3.1416 (R) | s[4]: -0.7646 (P) | s[5]: 1.5916 (Y) | s[6]: -0.0000 (f)   





    # --- 3. 应用约束并执行动作 ---
    target_pos[0] = np.clip(target_pos[0], 0.3, 0.8)
    target_pos[1] = np.clip(target_pos[1], -0.3, 0.3)
    target_pos[2] = np.clip(target_pos[2], 0.2, 0.7)
    finger_angle = np.clip(finger_angle, 0.0, 0.4)

    target_orn = p.getQuaternionFromEuler(target_euler)
    kuka.setInverseKine(target_pos, target_orn, finger_angle)

    # --- 4. 状态输出与同步 (已修改) ---
    current_obs = kuka.getObservation()
    
    # 将当前状态写入共享字典，供GUI读取
    with shared_state['lock']:
        shared_state['current_pos'] = list(current_obs[0:3])
        shared_state['current_euler'] = list(current_obs[3:6])

    # 实时打印 s[0] 到 s[6] 的状态到控制台
    s_subset = current_obs[0:7]
    output_string = (
        f"s[0]: {s_subset[0]:.4f} (x) | "
        f"s[1]: {s_subset[1]:.4f} (y) | "
        f"s[2]: {s_subset[2]:.4f} (z) | "
        f"s[3]: {s_subset[3]:.4f} (R) | "
        f"s[4]: {s_subset[4]:.4f} (P) | "
        f"s[5]: {s_subset[5]:.4f} (Y) | "
        f"s[6]: {s_subset[6]:.4f} (f)"
    )
    print(f"\r{output_string}   ", end="")


    # --- 5. 仿真步进 ---
    p.stepSimulation()
    time.sleep(TIME_STEP)

print("\n仿真结束。")
p.disconnect()