#!/usr/bin/env python3
"""
Unified dashboard entrypoint that wires together all Flask apps in this
workspace (camera stream, driver, GPIO knock, linear slider, and voice tools).

Usage:
    python dashboard.py --host 0.0.0.0 --port 5000 --debug

The implementation reuses the existing app factory exposed by voice.app, which
already registers the other blueprints and Socket.IO handlers.
"""

from __future__ import annotations

import argparse
from typing import Tuple

from flask import Flask
from flask_socketio import SocketIO


def create_dashboard() -> Tuple[Flask, SocketIO]:
    """
    Build the combined Flask application and its Socket.IO server.

    Returns:
        (app, socketio): Tuple containing the ready-to-run Flask app and the
        Socket.IO server instance.
    """
    from voice.app import create_app as create_voice_app
    from voice.app import create_socketio

    app = create_voice_app()
    socketio = create_socketio(app)
    return app, socketio


app, socketio = create_dashboard()


def main() -> None:
    """CLI entrypoint for running the unified dashboard."""
    parser = argparse.ArgumentParser(description="Run the robotics control dashboard.")
    parser.add_argument("--host", default="0.0.0.0", help="Host interface to bind.")
    parser.add_argument("--port", type=int, default=5000, help="TCP port to listen on.")
    parser.add_argument(
        "--debug",
        action="store_true",
        help="Enable Flask debug mode (Werkzeug reloader).",
    )
    args = parser.parse_args()

    socketio.run(
        app,
        host=args.host,
        port=args.port,
        debug=args.debug,
        allow_unsafe_werkzeug=True,
    )


if __name__ == "__main__":
    main()
