#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <sys/time.h>
#include "ACTS2200.h"

// Linux-friendly replacement for kbhit()
#include <termios.h>
#include <fcntl.h>
int kbhit(void) {
    struct termios oldt, newt;
    int ch;
    int oldf;
    tcgetattr(STDIN_FILENO, &oldt);
    newt = oldt;
    newt.c_lflag &= ~(ICANON | ECHO);
    tcsetattr(STDIN_FILENO, TCSANOW, &newt);
    oldf = fcntl(STDIN_FILENO, F_GETFL, 0);
    fcntl(STDIN_FILENO, F_SETFL, oldf | O_NONBLOCK);
    ch = getchar();
    tcsetattr(STDIN_FILENO, TCSANOW, &oldt);
    fcntl(STDIN_FILENO, F_SETFL, oldf);
    if (ch != EOF) {
        ungetc(ch, stdin);
        return 1;
    }
    return 0;
}

// 获取当前时间（秒.微秒）
double get_time_sec()
{
    struct timeval tv;
    gettimeofday(&tv, NULL);
    return tv.tv_sec + tv.tv_usec / 1e6;
}

int main(void)
{
    HANDLE hDevice;
    ACTS2200_AI_PARAM AIParam;
    double *dataBuf;
    U32 readCount;
    const U32 chanCount = 1;
    const U32 chunk = 1000;
    const double sampRate = 10000.0;
    FILE *fp;
    double startTime, curTime;
    unsigned long long totalPoints = 0;

    printf("Opening device...\n");
    hDevice = ACTS2200_DEV_Create(0);
    if (hDevice == INVALID_HANDLE_VALUE) {
        printf("Device open failed!\n");
        return -1;
    }

    memset(&AIParam, 0, sizeof(AIParam));
    AIParam.nSampChanCount = chanCount;
    AIParam.CHParam[0].nChannel   = 0;
    AIParam.CHParam[0].nSampRange = ACTS2200_AI_SAMPRANGE_N10_P10V;
    AIParam.CHParam[0].nRefGround = ACTS2200_AI_REFGND_RSE;
    AIParam.nSampMode     = ACTS2200_SAMPMODE_CONTINUOUS;
    AIParam.nSampsPerChan = 4096;
    AIParam.fSampRate     = sampRate;
    AIParam.nSampClkSrc   = ACTS2200_VAL_LOCAL;
    AIParam.nFillMode     = ACTS2200_FILLMODE_GroupByScanNumber;
    AIParam.StartTrig.nTrigType = ACTS2200_VAL_NONE;

    if (!ACTS2200_AI_InitTask(hDevice, &AIParam)) {
        printf("AI_InitTask failed!\n");
        ACTS2200_DEV_Release(hDevice);
        return -2;
    }

    if (!ACTS2200_AI_StartTask(hDevice)) {
        printf("AI_StartTask failed!\n");
        ACTS2200_AI_ReleaseTask(hDevice);
        ACTS2200_DEV_Release(hDevice);
        return -3;
    }

    printf("Start continuous acquisition...\n");
    printf("Sampling rate = %.1f Hz, chunk = %u points\n", sampRate, chunk);
    printf("Press any key to stop.\n");

    dataBuf = (double*)malloc(sizeof(double) * chanCount * chunk);
    if (!dataBuf) {
        printf("No memory!\n");
        ACTS2200_AI_StopTask(hDevice);
        ACTS2200_AI_ReleaseTask(hDevice);
        ACTS2200_DEV_Release(hDevice);
        return -4;
    }

    fp = fopen("/home/pi/acc_data/ai0_data.csv", "a");
    if (!fp) {
        printf("Cannot open /home/pi/acc_data/ai0_data.csv\n");
        free(dataBuf);
        ACTS2200_AI_StopTask(hDevice);
        ACTS2200_AI_ReleaseTask(hDevice);
        ACTS2200_DEV_Release(hDevice);
        return -5;
    }

    fseek(fp, 0, SEEK_END);
    if (ftell(fp) == 0)
        fprintf(fp, "Time(s),Voltage(V)\n");

    startTime = get_time_sec();

    while (!kbhit()) {
        if (!ACTS2200_AI_ReadAnalog(hDevice, dataBuf, chunk, &readCount, 2.0,
                                    ACTS2200_FILLMODE_GroupByScanNumber)) {
            printf("Read timeout or error.\n");
            continue;
        }

        curTime = get_time_sec();
        double elapsed = curTime - startTime;
        double timeStep = 1.0 / sampRate;

        for (U32 i = 0; i < readCount; ++i) {
            double t = elapsed + i * timeStep;
            fprintf(fp, "%.6f,%.6f\n", t, dataBuf[i]);
        }

        fflush(fp);
        totalPoints += readCount;
    }

    double endTime = get_time_sec();
    double totalSec = endTime - startTime;

    printf("\nStopping...\n");
    printf("Total points: %llu\n", totalPoints);
    printf("Total time: %.3f seconds\n", totalSec);
    printf("Approx. effective rate: %.1f Hz\n", totalPoints / totalSec);

    fclose(fp);
    free(dataBuf);
    ACTS2200_AI_StopTask(hDevice);
    ACTS2200_AI_ReleaseTask(hDevice);
    ACTS2200_DEV_Release(hDevice);
    printf("Done. Data saved to /home/pi/acc_data/ai0_data.csv\n");

    return 0;
}
